/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.
	
	Platform independent state machine based implementation of
	IETF RFC 2409: Internet Key Exchange protocol
	Also implements XAuth V6+, Config Mode, and NAT Traversal.

	$Id: pgpIKE.c,v 1.75.2.10 2001/07/23 01:11:56 wprice Exp $
____________________________________________________________________________*/

#include "pgpIKEPriv.h"

#include "pgpErrors.h"
#include "pgpContext.h"
#include "pgpMem.h"
#include "pgpEndianConversion.h"
#include "pgpHash.h"
#include "pgpHMAC.h"
#include "pgpPFLPriv.h"
#include "pgpPublicKey.h"
#include "pgpSymmetricCipher.h"
#include "pgpCBC.h"
#include "pgpKeys.h"
#include "pgpFeatures.h"
#include "pgpMilliseconds.h"

#include <string.h>
#include <stdio.h>

#if PGP_DEBUG
#define PGPIKE_DEBUG	1
#define PGPIKE_VERBOSE	1
#else
#define PGPIKE_DEBUG	1
#define PGPIKE_VERBOSE	0
#endif

const PGPikeTransform kPGPike_DefaultIKETransforms[] =
	{
		{
			kPGPike_AM_PreSharedKey,
			kPGPike_HA_SHA1,
			kPGPike_SC_CAST_CBC,
			TRUE, kPGPike_GR_MODPFive
		},
		{
			kPGPike_AM_PreSharedKey,
			kPGPike_HA_MD5,
			kPGPike_SC_3DES_CBC,
			TRUE, kPGPike_GR_MODPTwo
		},
		{
			kPGPike_AM_DSS_Sig,
			kPGPike_HA_SHA1,
			kPGPike_SC_CAST_CBC,
			TRUE, kPGPike_GR_MODPFive
		},
		{
			kPGPike_AM_DSS_Sig,
			kPGPike_HA_SHA1,
			kPGPike_SC_3DES_CBC,
			TRUE, kPGPike_GR_MODPTwo
		},
		{
			kPGPike_AM_RSA_Sig,
			kPGPike_HA_SHA1,
			kPGPike_SC_CAST_CBC,
			TRUE, kPGPike_GR_MODPFive
		},
		{
			kPGPike_AM_RSA_Sig,
			kPGPike_HA_MD5,
			kPGPike_SC_3DES_CBC,
			TRUE, kPGPike_GR_MODPTwo
		},
	};

const PGPipsecTransform kPGPike_DefaultIPSECTransforms[] =
	{
		{	/* ESP, CAST, SHA1 */
			TRUE, { kPGPike_ET_CAST, kPGPike_AA_HMAC_SHA, kPGPike_PM_None },
			FALSE, { kPGPike_AH_None, kPGPike_AA_None, kPGPike_PM_None },
			FALSE, { kPGPike_IC_None },
			kPGPike_GR_None
		},
		{	/* ESP, 3DES, MD5 */
			TRUE, { kPGPike_ET_3DES, kPGPike_AA_HMAC_MD5, kPGPike_PM_None },
			FALSE, { kPGPike_AH_None, kPGPike_AA_None, kPGPike_PM_None },
			FALSE, { kPGPike_IC_None },
			kPGPike_GR_None
		},
	};


const PGPByte				dhGenerator[1] = { 2 };
const char					kPGPike_PGPVendorString1[] = "OpenPGP1";
const char					kPGPike_PGPVendorString2[] = "0171";	/* last v change is SDK 171 */
const PGPByte				kPGPike_XAuth6VendorID[8] =
								{	0x09, 0x00, 0x26, 0x89, 0xDF, 0xD6, 0xB7, 0x12	};

/* This is the MD5 hash of "ESPThruNAT" from draft-huttunen-espinudp-01 */
const PGPByte				kPGPike_NATTraversalVendorID[16] =
								{	0x50, 0x76, 0x0f, 0x62, 0x4c, 0x63, 0xe5, 0xc5, 
									0x3e, 0xea, 0x38, 0x6c, 0x68, 0x5c, 0xa0, 0x83	};

    PGPError
PGPNewIKEContext(
	PGPContextRef			context,
	PGPikeMessageProcPtr	ikeMessageProc,
	void *					inUserData,
	PGPikeContextRef *		outRef )
{
	PGPError				err	= kPGPError_NoErr;
	PGPikeContextPriv *		pContext;

	*outRef = NULL;
	PGPValidatePtr( context );
	if( IsNull( ikeMessageProc ) )
		return kPGPError_BadParams;
	
	pContext = (PGPikeContextPriv *) pgpContextMemAlloc( context,
											sizeof(PGPikeContextPriv),
											kPGPMemoryMgrFlags_Clear );
	if( IsntNull( pContext ) )
	{
		pContext->pgpContext		= context;
		pContext->memMgr			= PGPPeekContextMemoryMgr( context );
		pContext->msgProc			= ikeMessageProc;
		pContext->userData			= inUserData;
		pContext->pending			= NULL;
		pContext->cookieDough		= FALSE;
		err = PGPContextGetRandomBytes( context, &pContext->cookieSecret,
									kPGPike_CookieSize );
		if( IsntPGPError( err ) )
			pContext->cookieDough = TRUE;
		err = kPGPError_NoErr;	/* no entropy error here will be retried later */
		
		pContext->secLifeTimeIKE	= kPGPike_DefaultSecLife;
		pContext->kbLifeTimeIKE		= kPGPike_DefaultKBLife;
		pContext->secLifeTimeIPSEC	= kPGPike_DefaultSecLife;
		pContext->kbLifeTimeIPSEC	= kPGPike_DefaultKBLife;
		
		pContext->defaultIKEProps = (PGPikeTransform *) pgpContextMemAlloc(
				context, sizeof(kPGPike_DefaultIKETransforms),
				kPGPMemoryMgrFlags_Clear );
		if( IsNull( pContext->defaultIKEProps ) )
		{
			err = kPGPError_OutOfMemory;
			goto done;
		}
		pContext->numIKEProps = sizeof(kPGPike_DefaultIKETransforms) /
								sizeof(PGPikeTransform);
		pgpCopyMemory( &kPGPike_DefaultIKETransforms[0], pContext->defaultIKEProps,
						sizeof(kPGPike_DefaultIKETransforms) );

		pContext->defaultIPSECProps = (PGPipsecTransform *) pgpContextMemAlloc(
				context, sizeof(kPGPike_DefaultIPSECTransforms),
				kPGPMemoryMgrFlags_Clear );
		if( IsNull( pContext->defaultIPSECProps ) )
		{
			err = kPGPError_OutOfMemory;
			goto done;
		}
		pContext->numIPSECProps = sizeof(kPGPike_DefaultIPSECTransforms) /
								sizeof(PGPipsecTransform);
		pgpCopyMemory( &kPGPike_DefaultIPSECTransforms[0],
						pContext->defaultIPSECProps,
						sizeof(kPGPike_DefaultIPSECTransforms) );
		
		pContext->allowedAlgorithms.cast5			= TRUE;
		pContext->allowedAlgorithms.aes				= TRUE;
		pContext->allowedAlgorithms.tripleDES		= TRUE;
		pContext->allowedAlgorithms.singleDES		= FALSE;
		pContext->allowedAlgorithms.espNULL			= FALSE;
		pContext->allowedAlgorithms.sha1			= TRUE;
		pContext->allowedAlgorithms.sha2_256		= TRUE;
		pContext->allowedAlgorithms.sha2_384		= TRUE;
		pContext->allowedAlgorithms.sha2_512		= TRUE;
		pContext->allowedAlgorithms.md5				= TRUE;
		pContext->allowedAlgorithms.noAuth			= FALSE;
		pContext->allowedAlgorithms.lzs				= TRUE;
		pContext->allowedAlgorithms.deflate			= TRUE;
		pContext->allowedAlgorithms.modpOne768		= TRUE;
		pContext->allowedAlgorithms.modpTwo1024		= TRUE;
		pContext->allowedAlgorithms.modpFive1536	= TRUE;
		
		*outRef = ( PGPikeContextRef ) pContext;
	}
	else
		err = kPGPError_OutOfMemory;
	
done:
	if( IsPGPError( err ) && IsntNull( *outRef ) )
	{
		pgpContextMemFree( context, pContext );
		*outRef = NULL;
	}
	pgpAssertErrWithPtr( err, *outRef );
	return err;
}

	PGPError
PGPFreeIKEContext(
	PGPikeContextRef		ref )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeContextPriv *		ike;
	
	PGPValidatePtr( ref );
	ike = (PGPikeContextPriv *)ref;
	
	if( IsntNull( ike->pending ) )
	{
		PGPikeDestination	*	walkD,
							*	nextD;
		
		pgpAssert( 0 );
		/* REPORT THIS ASSERTION AND SUBMIT LOG */
		for( walkD = ike->pending; IsntNull( walkD ); walkD = nextD )
		{
			nextD = walkD->nextD;
			(void)PGPFreeData( walkD );
		}
	}
	while( IsntNull( ike->partner ) )
		(void)pgpIKEFreePartner( ike->partner );
	while( IsntNull( ike->sa ) )
	{
		PGPipsecSA *	oldHead = ike->sa;
		
		ike->sa = oldHead->nextSA;
		(void)PGPFreeData( oldHead );
	}
	
	err = pgpContextMemFree( ike->pgpContext, ike->defaultIKEProps ); CKERR;
	err = pgpContextMemFree( ike->pgpContext, ike->defaultIPSECProps ); CKERR;
	err = pgpContextMemFree( ike->pgpContext, ref ); CKERR;
done:
	return err;
}

	PGPError
pgpIKEFreePartner(
	PGPikePartner *			partner )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner	*		cp,
					*		lp;
	PGPikePendingDest *		walkPend;

	if( partner->ready )
	{
		if( partner->virtualIP && partner->assignedIP )
		{
			PGPikeMTIdentity	mtIdentity;
			
			pgpClearMemory( &mtIdentity, sizeof(PGPikeMTIdentity) );
			mtIdentity.active		= FALSE;
			mtIdentity.ipAddress	= partner->ipAddress;
			mtIdentity.assignedIP	= partner->assignedIP;
			mtIdentity.assignedDNS	= partner->assignedDNS;
			mtIdentity.assignedWINS	= partner->assignedWINS;
			err = (partner->ike->msgProc)(
					(PGPikeContextRef)partner->ike, partner->ike->userData,
					kPGPike_MT_Identity, &mtIdentity );	CKERR;
			partner->assignedIP		= 0;
			partner->virtualIP		= FALSE;
		}
		err = pgpIKELocalAlert( partner->ike, partner->ipAddress,
				kPGPike_AL_None, kPGPike_IA_DeadPhase1SA, FALSE );
	}
	/* delink ourself */
	cp = partner->ike->partner;	/* get head of list */
	lp = NULL;
	while( IsntNull( cp ) )
	{
		if( cp == partner )
		{
			if( IsntNull( lp ) )
				lp->nextPartner = cp->nextPartner;
			else
				partner->ike->partner = cp->nextPartner;
			break;
		}
		lp = cp;
		cp = cp->nextPartner;
	}
	while( IsntNull( partner->exchanges ) )
	{
		err = pgpIKEFreeExchange( partner->exchanges );	CKERR;
	}
	if( IsntNull( partner->sharedKey ) )
	{
		err = PGPFreeData( partner->sharedKey ); CKERR;
	}
	if( IsntNull( partner->idData ) )
	{
		err = PGPFreeData( partner->idData ); CKERR;
	}
	if( PGPKeyDBRefIsValid( partner->pgpAuthKey.baseKeyDB ) )
	{
		err = PGPFreeKeyDB( partner->pgpAuthKey.baseKeyDB ); CKERR;
	}
	if( PGPKeyDBRefIsValid( partner->x509AuthKey.baseKeyDB ) )
	{
		err = PGPFreeKeyDB( partner->x509AuthKey.baseKeyDB ); CKERR;
	}
	if( IsntNull( partner->pgpAuthKey.pass ) )
	{
		err = PGPFreeData( partner->pgpAuthKey.pass ); CKERR;
	}
	if( IsntNull( partner->x509AuthKey.pass ) )
	{
		err = PGPFreeData( partner->x509AuthKey.pass ); CKERR;
	}
	walkPend = partner->pendList;
	while( IsntNull( partner->pendList ) )
	{
		walkPend = partner->pendList;
		partner->pendList = partner->pendList->nextDest;
		(void) PGPFreeData( walkPend );
	}
	if( PGPCBCContextRefIsValid( partner->cbc ) )
	{
		err = PGPFreeCBCContext( partner->cbc );	CKERR;
	}
	if( PGPKeyDBRefIsValid( partner->remoteKeyDB ) )
	{
		err = PGPFreeKeyDB( partner->remoteKeyDB );	CKERR;
	}
	
	err = PGPFreeData( partner );
done:
	return err;
}

	PGPError
pgpIKEFreeExchange(
	PGPikeExchange *		exchange )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeExchange	*	cn,
						*	ln;
	PGPikeProposal		*	cp,
						*	lp = NULL;

	if( IsNull( exchange ) )
		goto done;
	exchange->ike->numExchanges--;
	/* delink ourself */
	if( IsntNull( exchange->destination ) )
		(void)PGPFreeData( exchange->destination );
	cn = exchange->partner->exchanges;	/* get head of list */
	ln = NULL;
	while( IsntNull( cn ) )
	{
		if( cn == exchange )
		{
			if( IsntNull( ln ) )
				ln->nextExchange = cn->nextExchange;
			else
				exchange->partner->exchanges = cn->nextExchange;
			break;
		}
		ln = cn;
		cn = cn->nextExchange;
	}

	cp = exchange->proposals;
	while( IsntNull( cp ) )
	{
		lp = cp;
		cp = cp->nextProposal;
		(void)PGPFreeData( lp );
	}
	pgpIKEFreeConfigs( exchange );
	if( exchange->dhP != kPGPInvalidBigNumRef )
		(void) PGPFreeBigNum( exchange->dhP );
	if( exchange->dhG != kPGPInvalidBigNumRef )
		(void) PGPFreeBigNum( exchange->dhG );
	if( exchange->dhX != kPGPInvalidBigNumRef )
		(void) PGPFreeBigNum( exchange->dhX );
	if( exchange->dhYi != kPGPInvalidBigNumRef )
		(void) PGPFreeBigNum( exchange->dhYi );
	if( exchange->dhYr != kPGPInvalidBigNumRef )
		(void) PGPFreeBigNum( exchange->dhYr );
	if( IsntNull( exchange->gXY ) )
		(void) PGPFreeData( exchange->gXY );
	
	if( IsntNull( exchange->initSABody ) )
		(void)PGPFreeData( exchange->initSABody );
	if( IsntNull( exchange->idBody ) )
		(void)PGPFreeData( exchange->idBody );
	if( IsntNull( exchange->idRBody ) )
		(void)PGPFreeData( exchange->idRBody );
	if( IsntNull( exchange->lastPkt ) )
		(void)PGPFreeData( exchange->lastPkt );
	
	(void)PGPFreeData( exchange );
done:
	return err;
}

	void
pgpIKEFreeConfigs(
	PGPikeExchange *		exchange )
{
	PGPUInt32				configIndex;
	
	if( IsNull( exchange ) )
		return;
	
	for( configIndex = 0; configIndex < exchange->numConfigs; configIndex++ )
	{
		if( IsntNull( exchange->configs[configIndex].valueP ) )
		{
			(void) PGPFreeData( exchange->configs[configIndex].valueP );
			exchange->configs[configIndex].valueP = NULL;
		}
		pgpClearMemory( &exchange->configs[configIndex], sizeof(PGPikeConfigItem) );
	}
	exchange->numConfigs = 0;
}

	PGPError
PGPikeProcessMessage(
	PGPikeContextRef		ref,
	PGPikeMessageType		msg,
	void *					data )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeContextPriv *		ike;
	
	PGPValidatePtr( ref );
	ike = (PGPikeContextPriv *)ref;
	
	switch( msg )
	{
		case kPGPike_MT_Idle:
			err = pgpIKEIdle( ike );
			break;
		case kPGPike_MT_SARequest:
			err = pgpIKEHandleSARequest( ike, (PGPikeMTSASetup *) data );
			break;
		case kPGPike_MT_SARekey:
			err = pgpIKEHandleSAEvent( ike, FALSE, (PGPipsecSA *) data );
			break;
		case kPGPike_MT_SADied:
			err = pgpIKEHandleSAEvent( ike, TRUE, (PGPipsecSA *) data );
			break;
		case kPGPike_MT_AuthCheck:
			err = pgpIKEHandleAuthCheck( ike, (PGPikeMTAuthCheck *) data );
			break;
		case kPGPike_MT_Packet:
			err = pgpIKEHandlePacket( ike, (PGPikeMTPacket *) data );
#if PGPIKE_VERBOSE
		/*{
			PGPUInt32	saCount = 0;
			PGPipsecSA *	sa;
			
			for( sa = ike->sa; IsntNull( sa ); sa = sa->nextSA )
				saCount++;
			pgpIKEDebugLog( ike, "\tsaCount: %d\n", saCount );
		}*/
#endif
			break;
		case kPGPike_MT_Pref:
			err = pgpIKESetPref( ike, (PGPikeMTPref *) data );
			break;
		case kPGPike_MT_SAKillAll:
		{
			PGPipsecSA	*	sa,
						*	nextSA = NULL;
			PGPikePartner *	partner;
			
			for( sa = ike->sa; IsntNull( sa ); sa = nextSA )
			{
				nextSA = sa->nextSA;
				sa->activeIn = FALSE;
				if( pgpIKEFindSAPartner( ike, sa, TRUE, &partner ) )
				{
					err = pgpIKEKillSA( &partner, sa );
				}
				else
				{
					err = (ike->msgProc)(
							(PGPikeContextRef)ike, ike->userData,
							kPGPike_MT_SADied, sa );
					(void)pgpIKEFreeSA( ike, sa );
				}
				if( IsPGPError( err ) )
					break;
			}
			break;
		}
		case kPGPike_MT_PolicyCheck:
		case kPGPike_MT_SAEstablished:
		case kPGPike_MT_LocalPGPCert:
		case kPGPike_MT_LocalX509Cert:
		case kPGPike_MT_RemoteCert:
		case kPGPike_MT_Alert:
		default:
			err = kPGPError_ItemNotFound;/* these msgs are only sent FROM pgpIKE */
			break;
	}
	if( IsPGPError( err ) )
	{
		if( err == kPGPError_UserAbort )
			err = kPGPError_NoErr;
		else
		{
#if PGPIKE_DEBUG
			pgpIKEDebugLog( ike, FALSE, "\tPGPError: %ld\n", err );
#endif
		}
	}
	return err;
}

	PGPBoolean
pgpIKEFindSAPartner(
	PGPikeContextPriv *		ike,
	PGPipsecSA *			sa,
	PGPBoolean				mustBeReady,
	PGPikePartner **		partnerP )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner;
	
	*partnerP = NULL;
	for( partner = ike->partner; IsntNull( partner );
			partner = partner->nextPartner )
	{
		if( ( sa->ipAddress == partner->ipAddress ) &&
			( partner->ready || !mustBeReady ) )
		{
			*partnerP = partner;
			goto done;
		}
	}
	err = kPGPError_ItemNotFound;
done:
	return ( err != kPGPError_ItemNotFound );
}

	PGPError
pgpIKEFreeSA(
	PGPikeContextPriv *		ike,
	PGPipsecSA *			sa )
{	
	PGPError				err = kPGPError_NoErr;
	
	err = pgpIKELocalAlert( ike, sa->ipAddress,
			kPGPike_AL_None, kPGPike_IA_DeadPhase2SA, FALSE );
	if( IsntNull( sa->prevSA ) )
		sa->prevSA->nextSA = sa->nextSA;
	else
		ike->sa = sa->nextSA;
	if( IsntNull( sa->nextSA ) )
		sa->nextSA->prevSA = sa->prevSA;
	(void)PGPFreeData( sa );
	return err;
}

	PGPError
pgpIKESetConfigData(
	PGPMemoryMgrRef			memMgr,
	PGPikeConfigItem *		config,
	PGPikeConfigType		configType,
	PGPikeAttributeType		attrType,
	PGPUInt16				transactionID,
	char *					configData )
{
	PGPError				err = kPGPError_NoErr;
	PGPUInt16				replyLen;

	replyLen				= strlen( configData );
	config->configType		= configType;
	config->transactionID	= transactionID;
	config->attribute		= attrType;
	config->valueP = PGPNewSecureData( memMgr, replyLen, 0 );
	if( IsntNull( config->valueP ) )
		pgpCopyMemory( configData, config->valueP, replyLen );
	else
		err = kPGPError_OutOfMemory;
	config->valueSize = replyLen;
	return err;
}

	PGPError
pgpIKEHandleAuthCheck(
	PGPikeContextPriv *		ike,
	PGPikeMTAuthCheck *		authCheck )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner;
	PGPikeExchange *		exchange;
	PGPHashContextRef		hash = kInvalidPGPHashContextRef;
	
	for( partner = ike->partner; IsntNull( partner ); partner = partner->nextPartner )
	{
		if( partner->ipAddress == authCheck->gatewayIP )
		{
			for( exchange = partner->exchanges; IsntNull( exchange );
					exchange = exchange->nextExchange )
			{
				if( ( exchange->exchangeT == kPGPike_EX_Transaction ) &&
					( exchange->numConfigs > 0 ) &&
					( exchange->configs[0].transactionID == authCheck->transactionID ) )
				{
					pgpIKEFreeConfigs( exchange );
					if( authCheck->success )
					{
						if( authCheck->includeType )
						{
							exchange->configs[0].configType		= kPGPike_CT_Reply;
							exchange->configs[0].transactionID	= authCheck->transactionID;
							exchange->configs[0].attribute		= kPGPike_AT_XAuth_Type;
							exchange->configs[0].value			= authCheck->xauthType;
							exchange->numConfigs++;
							switch( authCheck->xauthType )
							{
								case kPGPike_XT_RADIUS_CHAP:
								{
									PGPUInt32	challengeSize = authCheck->challengeSize;
									
									if( authCheck->useChallenge &&
										( challengeSize % 2 == 0 ) &&
										( challengeSize > 4 ) &&
										( challengeSize < 66 ) )
									{
										PGPByte				chapHash[16],
															authID;
										
										err = PGPContextGetRandomBytes( ike->pgpContext, &authID, 1);	CKERR;
										err = PGPNewHashContext( ike->pgpContext,
												kPGPHashAlgorithm_MD5, &hash );							CKERR;
										err = PGPContinueHash( hash, &authID, 1 );						CKERR;
										err = PGPContinueHash( hash, authCheck->password,
												strlen( authCheck->password ) );						CKERR;
										err = PGPContinueHash( hash, authCheck->challenge,
												challengeSize );										CKERR;
										err = PGPFinalizeHash( hash, chapHash );						CKERR;
										authCheck->challenge[0] = authID;
										pgpCopyMemory( chapHash, &authCheck->challenge[1], challengeSize );
										authCheck->challengeSize = challengeSize + 1;
									}
									else
									{
#if PGPIKE_DEBUG
										pgpIKEDebugLog( ike, TRUE, "\tBuggy gateway sent RADIUS_CHAP without Challenge\n" );
#endif
									}
									break;
								}
								case kPGPike_XT_OTP:
									goto authCheckFailed;
								case kPGPike_XT_SKEY:		/* we really should fail here, but Altiga bug sends this */
								default:
									break;
							}
						}
						if( authCheck->useUserName )
						{
							err = pgpIKESetConfigData( ike->memMgr, &exchange->configs[exchange->numConfigs],
									kPGPike_CT_Reply, kPGPike_AT_XAuth_Username, authCheck->transactionID,
									authCheck->userName ); CKERR;
							exchange->numConfigs++;
						}
						if( authCheck->usePassword )
						{
							if( authCheck->useChallenge )
							{
								exchange->configs[exchange->numConfigs].configType		= kPGPike_CT_Reply;
								exchange->configs[exchange->numConfigs].transactionID	= authCheck->transactionID;
								exchange->configs[exchange->numConfigs].attribute		= kPGPike_AT_XAuth_Password;
								exchange->configs[exchange->numConfigs].valueP =
									PGPNewSecureData( ike->memMgr, authCheck->challengeSize, 0 );
								if( IsntNull( exchange->configs[exchange->numConfigs].valueP ) )
									pgpCopyMemory(	authCheck->challenge,
													exchange->configs[exchange->numConfigs].valueP,
													authCheck->challengeSize );
								else
									err = kPGPError_OutOfMemory;
								exchange->configs[exchange->numConfigs].valueSize = authCheck->challengeSize;
								exchange->numConfigs++;
							}
							else
							{
								err = pgpIKESetConfigData( ike->memMgr, &exchange->configs[exchange->numConfigs],
										kPGPike_CT_Reply, kPGPike_AT_XAuth_Password, authCheck->transactionID,
										authCheck->password ); CKERR;
								exchange->numConfigs++;
							}
						}
						if( authCheck->usePasscode )
						{
							err = pgpIKESetConfigData( ike->memMgr, &exchange->configs[exchange->numConfigs],
									kPGPike_CT_Reply, kPGPike_AT_XAuth_Passcode, authCheck->transactionID,
									authCheck->passcode ); CKERR;
							exchange->numConfigs++;
						}
						if( authCheck->useDomain )
						{
							err = pgpIKESetConfigData( ike->memMgr, &exchange->configs[exchange->numConfigs],
									kPGPike_CT_Reply, kPGPike_AT_XAuth_Domain, authCheck->transactionID,
									authCheck->domain ); CKERR;
							exchange->numConfigs++;
						}
						if( exchange->numConfigs )
						{
							err = pgpIKEDoPacket( &exchange,
										kPGPike_PY_Attribute,
										kPGPike_PY_None );	CKERR;
#if !PGPIKE_XAUTHCISCO && !PGPIKE_XAUTHLUCENT
							pgpIKEFreeExchange( exchange );
#endif
						}
					}
					else
					{
						/*exchange->configs[0].configType		= kPGPike_CT_Set;
						exchange->configs[0].transactionID	= authCheck->transactionID;
						exchange->configs[0].attribute		= kPGPike_AT_XAuth_Status;
						exchange->configs[0].value			= 0;
						exchange->numConfigs = 1;*/
					authCheckFailed:
						exchange->partner->termSchedule = PGPGetTime();
						err = pgpIKEAbortExchange( &exchange, kPGPike_AL_AuthenticationFailed );
						goto done;
					}
					break;
				}
			}
			if( IsNull( exchange ) )
				err = kPGPError_ItemNotFound;
			break;
		}
	}
	if( IsNull( partner ) )
		err = kPGPError_ItemNotFound;
done:
	if( PGPHashContextRefIsValid( hash ) )
		(void)PGPFreeHashContext( hash );
	return err;
}

	PGPError
pgpIKEHandleSAEvent(
	PGPikeContextPriv *		ike,
	PGPBoolean				death,
	PGPipsecSA *			sa )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner;
	PGPikeExchange *		exchange;
	
	if( pgpIKEFindSAPartner( ike, sa, TRUE, &partner ) )
	{
		if( death )
		{
			err = pgpIKEKillSA( &partner, sa );	CKERR;
		}
		else if( IsNull( partner->exchanges ) )
		{
			PGPipsecSA *	oSA;
			PGPBoolean		found = FALSE;
			
			/* Make sure there isn't already another one exactly like it */
			for( oSA = ike->sa; IsntNull( oSA ); oSA = oSA->nextSA )
			{
				if( ( oSA->ipAddress == sa->ipAddress ) && ( sa != oSA ) &&
					( oSA->ipAddrStart == sa->ipAddrStart ) &&
					( oSA->ipMaskEnd == sa->ipMaskEnd ) &&
					( oSA->destIsRange == sa->destIsRange ) &&
					( oSA->ipPort == sa->ipPort ) &&
					( oSA->ipProtocol == sa->ipProtocol ) )
				{
					found = TRUE;
					break;
				}
			}
			if( found )
			{
#if PGPIKE_DEBUG
				pgpIKEDebugLog( ike, TRUE, "Remote side is not rekeying correctly. Compensating.\n" );
#endif
			}
			{
				PGPikePendingDest *	rekeyDest;
				
				rekeyDest = (PGPikePendingDest *)PGPNewData( ike->memMgr,
								sizeof(PGPikePendingDest), kPGPMemoryMgrFlags_Clear );
				if( IsNull( rekeyDest ) )
				{
					err = kPGPError_OutOfMemory;
					goto done;
				}
				rekeyDest->ipAddrStart	= sa->ipAddrStart;
				rekeyDest->ipMaskEnd	= sa->ipMaskEnd;
				rekeyDest->destIsRange	= sa->destIsRange;
				rekeyDest->ipPort		= sa->ipPort;
				rekeyDest->ipProtocol	= sa->ipProtocol;
				rekeyDest->nextDest		= partner->pendList;
				partner->pendList		= rekeyDest;
				err = pgpIKEStartQMExchange( partner, &exchange );	CKERR;
			}
		}
	}
	else if( death )
	{
#if PGPIKE_DEBUG
		pgpIKEDebugLog( ike, TRUE, "Kill P2 SA requested, no Phase 1 available!\n" );
#endif
		err = (ike->msgProc)( (PGPikeContextRef)ike,
				ike->userData, kPGPike_MT_SADied, sa );				CKERR;
		err = pgpIKEFreeSA( ike, sa );								CKERR;
	}
	else
	{
#if PGPIKE_DEBUG
		pgpIKEDebugLog( ike, TRUE, "Rekey requested, but no Phase 1 available!\n" );
#endif
	}
done:
	return err;
}

	PGPError
pgpIKESetPref(
	PGPikeContextPriv *		ike,
	PGPikeMTPref *			pref )
{
	PGPError				err = kPGPError_NoErr;
	
	switch( pref->pref )
	{
		case kPGPike_PF_Expiration:
			ike->secLifeTimeIKE			= pref->u.expiration.secLifeTimeIKE;
			if( ike->secLifeTimeIKE && ( ike->secLifeTimeIKE < 60 ) )
				ike->secLifeTimeIKE = 60;
			ike->kbLifeTimeIKE			= pref->u.expiration.kbLifeTimeIKE;
			if( ike->kbLifeTimeIKE && ( ike->kbLifeTimeIKE < kPGPike_KBLifeMinimum ) )
				ike->kbLifeTimeIKE = kPGPike_KBLifeMinimum;
			ike->secLifeTimeIPSEC		= pref->u.expiration.secLifeTimeIPSEC;
			if( ike->secLifeTimeIPSEC && ( ike->secLifeTimeIPSEC < 60 ) )
				ike->secLifeTimeIPSEC = 60;
			ike->kbLifeTimeIPSEC		= pref->u.expiration.kbLifeTimeIPSEC;
			if( ike->kbLifeTimeIPSEC && ( ike->kbLifeTimeIPSEC < kPGPike_KBLifeMinimum ) )
				ike->kbLifeTimeIPSEC = kPGPike_KBLifeMinimum;
			break;
		case kPGPike_PF_IKEProposals:
			if( IsntNull( ike->defaultIKEProps ) )
			{
				err = pgpContextMemFree( ike->pgpContext,
										ike->defaultIKEProps );CKERR;
			}
			ike->defaultIKEProps = (PGPikeTransform *)
				pgpContextMemAlloc( ike->pgpContext,
				sizeof(PGPikeTransform) * pref->u.ikeProposals.numTransforms,
				kPGPMemoryMgrFlags_Clear );
			if( IsNull( ike->defaultIKEProps ) )
			{
				err = kPGPError_OutOfMemory;
				goto done;
			}
			ike->numIKEProps = pref->u.ikeProposals.numTransforms;
			pgpCopyMemory( pref->u.ikeProposals.t, ike->defaultIKEProps,
							ike->numIKEProps * sizeof(PGPikeTransform) );
			break;
		case kPGPike_PF_IPSECProposals:
			if( IsntNull( ike->defaultIPSECProps ) )
			{
				err = pgpContextMemFree( ike->pgpContext,
										ike->defaultIPSECProps );CKERR;
			}
			ike->defaultIPSECProps = (PGPipsecTransform *)
				pgpContextMemAlloc( ike->pgpContext,
				sizeof(PGPipsecTransform) * pref->u.ipsecProposals.numTransforms,
				kPGPMemoryMgrFlags_Clear );
			if( IsNull( ike->defaultIPSECProps ) )
			{
				err = kPGPError_OutOfMemory;
				goto done;
			}
			ike->numIPSECProps = pref->u.ipsecProposals.numTransforms;
			pgpCopyMemory( pref->u.ipsecProposals.t, ike->defaultIPSECProps,
							ike->numIPSECProps * sizeof(PGPipsecTransform) );
			break;
		case kPGPike_PF_AllowedAlgorithms:
			pgpCopyMemory( &pref->u.allowedAlgorithms, &ike->allowedAlgorithms,
							sizeof(PGPikeAllowedAlgorithms) );
			break;
		default:
			err = kPGPError_FeatureNotAvailable;
			break;
	}
done:
	return err;
}

	PGPError
pgpIKEAddPending(
	PGPikeContextPriv *		ike,
	PGPUInt32				ipAddress,
	PGPUInt32				ipAddrStart,
	PGPUInt32				ipMaskEnd,
	PGPBoolean				destIsRange )
{
	PGPikeDestination *		dest;
	PGPError				err = kPGPError_NoErr;
	
	/* The pending list is really just for debugging purposes */
	dest = (PGPikeDestination *)PGPNewData( ike->memMgr, sizeof(PGPikeDestination), 0 );
	if( IsntNull( dest ) )
	{
		dest->ipAddress		= ipAddress;
		dest->ipAddrStart	= ipAddrStart;
		dest->ipMaskEnd		= ipMaskEnd;
		dest->destIsRange	= destIsRange;
		dest->nextD			= ike->pending;
		ike->pending		= dest;
	}
	else
		err = kPGPError_OutOfMemory;
	return err;
}

	PGPBoolean
pgpIKEFindPending(
	PGPikeContextPriv *		ike,
	PGPUInt32				ipAddress,
	PGPUInt32				ipAddrStart,
	PGPUInt32				ipMaskEnd,
	PGPBoolean				destIsRange )
{
	PGPBoolean				found = FALSE;
	PGPikeDestination *		dest;
	
	for( dest = ike->pending; IsntNull( dest ); dest = dest->nextD )
	{
		if( ( dest->ipAddress == ipAddress ) &&
			( dest->ipAddrStart == ipAddrStart ) &&
			( dest->ipMaskEnd == ipMaskEnd ) &&
			( dest->destIsRange == destIsRange ) )
		{
			found = TRUE;
			break;
		}
	}
	return found;
}

	PGPError
pgpIKERemovePending(
	PGPikeContextPriv *		ike,
	PGPUInt32				ipAddress,
	PGPUInt32				ipAddrStart,
	PGPUInt32				ipMaskEnd,
	PGPBoolean				destIsRange )
{
	PGPikeDestination *		dest;
	PGPikeDestination *		lastDest = NULL;
	PGPError				err = kPGPError_NoErr;
	
	for( dest = ike->pending; IsntNull( dest ); dest = dest->nextD )
	{
		if( ( dest->ipAddress == ipAddress ) &&
			( dest->ipAddrStart == ipAddrStart ) &&
			( dest->ipMaskEnd == ipMaskEnd ) &&
			( dest->destIsRange == destIsRange ) )
		{
			if( IsNull( lastDest ) )
				ike->pending = dest->nextD;
			else
				lastDest->nextD = dest->nextD;
			(void)PGPFreeData( dest );
			break;
		}
		lastDest = dest;
	}
	return err;
}

	PGPError
pgpIKEHandleSARequest(
	PGPikeContextPriv *		ike,
	PGPikeMTSASetup *		saReq )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner = NULL;
	PGPipsecSA *			sa = NULL;
	PGPBoolean				found = FALSE,
							foundSA = FALSE;

	if( IsNull( saReq ) )
		return kPGPError_BadParams;
	
#if PGPIKE_DEBUG
	{
		char	ip1Str[20],
				ip2Str[20],
				ip3Str[20];
		
		pgpIKEGetIPString( saReq->ipAddress, ip1Str );
		pgpIKEGetIPString( saReq->ipAddrStart, ip2Str );
		pgpIKEGetIPString( saReq->ipMaskEnd, ip3Str );
		pgpIKEDebugLog( ike, TRUE, "SARequest: %s (%s/%s)\n", ip1Str, ip2Str, ip3Str );
	}
#endif
	if( pgpIKEFindPending(  ike, saReq->ipAddress, saReq->ipAddrStart,
			saReq->ipMaskEnd, saReq->destIsRange ) )
	{
#if PGPIKE_DEBUG
		pgpIKEDebugLog( ike, TRUE, "SARequest: IGNORED!! Already in progress.\n" );
#endif
		goto done;
	}
	err = pgpIKEAddPending( ike, saReq->ipAddress, saReq->ipAddrStart,
			saReq->ipMaskEnd, saReq->destIsRange );	CKERR;
	for( partner = ike->partner; IsntNull( partner ); partner = partner->nextPartner )
	{
		if( ( partner->ipAddress == saReq->ipAddress ) && partner->ready )
		{
			/* We only re-use an existing P1 if it has SAs available */
			for( sa = ike->sa; IsntNull( sa ); sa = sa->nextSA )
			{
				if( sa->ipAddress == saReq->ipAddress )
				{
					foundSA = TRUE;
					break;
				}
			}
			if( foundSA )
			{
				PGPikePendingDest *	pendDest;
				
				found = TRUE;
				pendDest = PGPNewData( ike->memMgr,
									sizeof(PGPikePendingDest),
									kPGPMemoryMgrFlags_Clear );
				if( IsNull( pendDest ) )
				{
					err = kPGPError_OutOfMemory;
					goto done;
				}
				pendDest->ipAddrStart	= saReq->ipAddrStart;
				pendDest->ipMaskEnd		= saReq->ipMaskEnd;
				pendDest->destIsRange	= saReq->destIsRange;
				pendDest->ipPort		= saReq->ipPort;
				pendDest->ipProtocol	= saReq->ipProtocol;
				pendDest->nextDest		= partner->pendList;
				partner->pendList		= pendDest;
#if PGPIKE_DEBUG
				pgpIKEDebugLog( ike, TRUE, "SARequest: P1 Already exists, going to P2\n" );
#endif
			}
		}
	}
	if( !foundSA )
		err = pgpIKEStartIdentityExchange( ike, saReq->ipAddress, saReq );	CKERR;
done:
	return err;
}

	PGPError
pgpIKECreateProposals(
	PGPikeExchange *		exchange )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeProposal *		proposal = NULL;
	PGPUInt16				transIndex,
							numTransforms;

	proposal = PGPNewData( exchange->ike->memMgr, sizeof(PGPikeProposal),
								kPGPMemoryMgrFlags_Clear );
	if( IsNull( proposal ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}
	switch( exchange->exchangeT )
	{
		case kPGPike_EX_Identity:
		case kPGPike_EX_Aggressive:
			proposal->number				= 1;
			proposal->protocol				= kPGPike_PR_IKE;
			numTransforms = exchange->ike->numIKEProps;
			if( numTransforms > kPGPike_MaxPropTransforms )
				numTransforms = kPGPike_MaxPropTransforms;
			for( transIndex = 0; transIndex < numTransforms; transIndex++ )
			{
				PGPikeTransform		xTransform;
			
				pgpCopyMemory(	&exchange->ike->defaultIKEProps[transIndex], &xTransform,
								sizeof( PGPikeTransform ) );
				if( exchange->partner->authStyle != kPGPike_AS_Normal )
				{
#if !PGPIKE_XAUTHCISCO
					if( exchange->partner->authStyle == kPGPike_AS_XAuth )
					{
						switch( xTransform.authMethod )
						{
							case kPGPike_AM_PreSharedKey:
								xTransform.authMethod = kPGPike_AM_XAuth_InitPreShared;
								break;
							case kPGPike_AM_DSS_Sig:
								xTransform.authMethod = kPGPike_AM_XAuth_InitDSS;
								break;
							case kPGPike_AM_RSA_Sig:
								xTransform.authMethod = kPGPike_AM_XAuth_InitRSA;
								break;
							default:
								continue;
						}
					}
					else if( exchange->partner->authStyle == kPGPike_AS_HybridAuth )
					{
						switch( xTransform.authMethod )
						{
							case kPGPike_AM_DSS_Sig:
								xTransform.authMethod = kPGPike_AM_HAuth_InitDSS;
								break;
							case kPGPike_AM_RSA_Sig:
								xTransform.authMethod = kPGPike_AM_HAuth_InitRSA;
								break;
							default:
								continue;
						}
					}
#endif
				}
				if( pgpIKEIsTransformValid( exchange->partner, kPGPike_PR_IKE,
						(PGPikeGenericTransform *)&xTransform ) )
				{
					pgpCopyMemory(	&xTransform, &proposal->t[proposal->numTransforms].u.ike,
									sizeof( PGPikeTransform ) );
					proposal->numTransforms++;
				}
				if( proposal->numTransforms &&
					( exchange->exchangeT == kPGPike_EX_Aggressive ) )
					break;
			}
			proposal->secLifeTime			= exchange->ike->secLifeTimeIKE;
			proposal->kbLifeTime			= exchange->ike->kbLifeTimeIKE;
			break;
		case kPGPike_EX_IPSEC_Quick:
		{
			PGPikeProposal	*	nproposal = NULL;
			PGPUInt32			propType,
								propIndex;
			PGPBoolean			useProtocol;
			PGPipsecTransform *	curp;
			
			pgpAssert( IsntNull( exchange->destination ) );
			for( propIndex = exchange->ike->numIPSECProps; propIndex > 0; propIndex-- )
			{
				curp = &exchange->ike->defaultIPSECProps[propIndex - 1];
				for( propType = 0; propType < 3; propType++ )
				{
					useProtocol = FALSE;
					if( IsNull( nproposal ) )
						nproposal = proposal;
					else
					{
						nproposal = PGPNewData( exchange->ike->memMgr,
							sizeof(PGPikeProposal), kPGPMemoryMgrFlags_Clear );
						if( IsNull( nproposal ) )
						{
							err = kPGPError_OutOfMemory;
							goto done;
						}
						nproposal->nextProposal = proposal;
						proposal = nproposal;
					}
					switch( propType )	/* in reverse order */
					{
						case 0:			/* IPCOMP */
							if( curp->useIPCOMP )
							{
								useProtocol = TRUE;
								proposal->protocol	= kPGPike_PR_IPCOMP;
								pgpCopyMemory( &curp->ipcomp,
												&proposal->t[0].u.ipsec.ipcomp,
												sizeof(PGPipsecIPCOMPTransform) );
							}
							break;
						case 1:			/* ESP */
							if( curp->useESP )
							{
								useProtocol = TRUE;
								proposal->protocol	= kPGPike_PR_ESP;
								pgpCopyMemory( &curp->esp, &proposal->t[0].u.ipsec.esp,
												sizeof(PGPipsecESPTransform) );
								proposal->t[0].u.ipsec.esp.mode =
									( exchange->destination->ipAddrStart > 0 ) ?
										kPGPike_PM_Tunnel : kPGPike_PM_Transport;
							}
							break;
						case 2:			/* AH */
							if( curp->useAH )
							{
								useProtocol = TRUE;
								proposal->protocol	= kPGPike_PR_AH;
								pgpCopyMemory( &curp->ah, &proposal->t[0].u.ipsec.ah,
												sizeof(PGPipsecAHTransform) );
								proposal->t[0].u.ipsec.ah.mode =
									( exchange->destination->ipAddrStart > 0 ) ?
										kPGPike_PM_Tunnel : kPGPike_PM_Transport;
							}
							break;
					}
					if( useProtocol )
					{
						proposal->number		= propIndex;
						proposal->numTransforms	= 1;
						proposal->t[0].u.ipsec.groupID	= curp->groupID;
						proposal->secLifeTime	= exchange->ike->secLifeTimeIPSEC;
						proposal->kbLifeTime	= exchange->ike->kbLifeTimeIPSEC;
						err = PGPContextGetRandomBytes( exchange->ike->pgpContext,
									&proposal->initSPI, sizeof(PGPipsecSPI) ); CKERR;
					}
					else
					{
						nproposal = proposal;
						proposal = proposal->nextProposal;
						err = PGPFreeData( nproposal );	CKERR;
					}
				}
			}
			break;
		}
		default:
			break;
	}
	pgpAssert( IsNull( exchange->proposals ) );
	exchange->proposals = proposal;
	proposal = NULL;
done:
	if( IsntNull( proposal ) )
		(void) PGPFreeData( proposal );
	return err;
}

	PGPError
pgpIKECreateExchange(
	PGPikePartner *			partner,
	PGPikeExchangeT			exchangeT,
	PGPikeMessageID *		messageID,
	PGPikeState				state,
	PGPBoolean				initiator,
	PGPikeExchange **		exchangeP )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeExchange *		exchange = NULL;
	PGPHashContextRef		hash = kInvalidPGPHashContextRef;

	*exchangeP = NULL;
	pgpAssertAddrValid( partner, PGPikePartner );
	
	exchange = PGPNewData( partner->ike->memMgr, sizeof(PGPikeExchange),
							kPGPMemoryMgrFlags_Clear );
	if( IsNull( exchange ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}
	partner->ike->numExchanges++;
	exchange->ike					= partner->ike;
	exchange->partner				= partner;
	exchange->initiator				= initiator;
	exchange->state					= state;
	exchange->doi					= kPGPike_DOI_IPSEC;
	exchange->exchangeT				= exchangeT;
	
#if PGPIKE_DEBUG
	pgpIKEDebugLog( partner->ike, TRUE, "\tNew %s Exchange - %s\n",
		pgpIKEExchangeTypeString( exchangeT ),
		initiator ? "Initiator" : "Responder" );
#endif
	if( ( exchangeT == kPGPike_EX_IPSEC_Quick ) ||
		( ( ( exchangeT == kPGPike_EX_Informational ) ||
			( exchangeT == kPGPike_EX_Transaction ) )
			&& partner->ready ) )
	{
		PGPByte		p2iv[kPGPike_MaxHashSize];
		
		if( IsNull( messageID ) )
		{
			err = PGPContextGetRandomBytes( partner->ike->pgpContext,
									&exchange->messageID,
									sizeof(PGPikeMessageID) ); CKERR;
		}
		else
		{
			pgpCopyMemory( messageID, exchange->messageID,
							sizeof(PGPikeMessageID) );
		}
		if( exchangeT == kPGPike_EX_IPSEC_Quick )
		{
			pgpCopyMemory( exchange->messageID, partner->oldMessageID[partner->oldMessageIDIndex],
				sizeof(PGPikeMessageID) );
			if( ++partner->oldMessageIDIndex >= kPGPike_MaxOldExchanges )
				partner->oldMessageIDIndex = 0;
		}
		
#if PGPIKE_VERBOSE
		pgpIKEDebugData( partner->ike, "\tMessageID",
			exchange->messageID, sizeof(PGPikeMessageID) );
#endif
		/* Set the last CBC block to correct value for a new P1/P2 exchange */
		err = PGPNewHashContext( partner->ike->pgpContext,
						partner->sdkHashAlg, &hash );	CKERR;
		err = PGPContinueHash( hash, partner->lastP1CBC,
								kPGPike_MaxCipherBlockSize );	CKERR;
		err = PGPContinueHash( hash, &exchange->messageID,
								sizeof(PGPikeMessageID) );	CKERR;
		err = PGPFinalizeHash( hash, p2iv );	CKERR;
		pgpCopyMemory( p2iv, exchange->lastCBC, partner->cipherBlockSize );
	}
	
	if( IsNull( partner->exchanges ) )
		partner->exchanges	= exchange;
	else
	{
		PGPikeExchange *cn = partner->exchanges;
		
		while( IsntNull( cn->nextExchange ) )
			cn = cn->nextExchange;
		cn->nextExchange		= exchange;
	}
	*exchangeP = exchange;
done:
	if( PGPHashContextRefIsValid( hash ) )
		(void)PGPFreeHashContext( hash );
	if( IsPGPError( err ) )
		(void)PGPFreeData( exchange );
	return err;
}

	PGPError
pgpIKECreatePartner(
	PGPikeContextPriv *		ike,
	PGPikeMTSASetup *		setup,
	PGPBoolean				initiator,
	PGPikePartner **		partnerP )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner;

	pgpAssert( IsntNull( setup ) );
	pgpAssert( IsntNull( partnerP ) );
	*partnerP = NULL;
	partner = PGPNewData( ike->memMgr, sizeof(PGPikePartner),
							kPGPMemoryMgrFlags_Clear );
	if( IsntNull( partner ) )
	{
		partner->magic				= kPGPike_Magic;
		partner->rttMillisec		= kPGPike_RoundTripSlack * 2;
		partner->ike				= ike;
		partner->ipAddress			= setup->ipAddress;
		partner->localIPAddress		= setup->localIPAddress;
		partner->initiator			= initiator;
		partner->destPort			= kPGPike_CommonPort;
		partner->virtualIP			= setup->virtualIP;
		partner->authStyle			= setup->authStyle;
		partner->bNATTraversal		= setup->forceEncapsulate;
		partner->birthTime			= PGPGetTime();
		if( initiator )
			err = pgpIKEBakeCookie( ike, setup->ipAddress, setup->localIPAddress,
									&partner->initCookie[0] );
		else
			err = pgpIKEBakeCookie( ike, setup->ipAddress, setup->localIPAddress,
									&partner->respCookie[0] );
		CKERR;
		partner->sharedKeySize = setup->sharedKeySize;
		if( IsntNull( setup->sharedKey ) )
		{
			partner->sharedKey = PGPNewSecureData( ike->memMgr,
					setup->sharedKeySize, kPGPMemoryMgrFlags_Clear );
			if( IsntNull( partner->sharedKey ) )
				pgpCopyMemory( setup->sharedKey,
								partner->sharedKey,
								setup->sharedKeySize );
			else
			{
				err = kPGPError_OutOfMemory;
				goto done;
			}
		}
		if( IsntNull( setup->idData ) )
		{
			partner->idData = PGPNewData( ike->memMgr,
								setup->idDataSize,
								kPGPMemoryMgrFlags_Clear );
			if( IsNull( partner->idData ) )
			{
				err = kPGPError_OutOfMemory;
				goto done;
			}
			partner->idDataSize = setup->idDataSize;
			partner->idType = setup->idType;
			pgpCopyMemory( setup->idData, partner->idData, partner->idDataSize );
		}
		if( initiator )
		{
			partner->pendList = PGPNewData( ike->memMgr,
								sizeof(PGPikePendingDest),
								kPGPMemoryMgrFlags_Clear );
			if( IsNull( partner->pendList ) )
			{
				err = kPGPError_OutOfMemory;
				goto done;
			}
			partner->pendList->ipAddrStart	= setup->ipAddrStart;
			partner->pendList->ipMaskEnd	= setup->ipMaskEnd;
			partner->pendList->destIsRange	= setup->destIsRange;
			partner->pendList->ipPort		= setup->ipPort;
			partner->pendList->ipProtocol	= setup->ipProtocol;
		}
		if( partner->authStyle != kPGPike_AS_HybridAuth )
		{
			err = pgpIKEGetCert( partner, kPGPike_MT_LocalPGPCert ); CKERR;
			err = pgpIKEGetCert( partner, kPGPike_MT_LocalX509Cert ); CKERR;
		}
				
		/* we're all set, hook it into our partners list */
		partner->nextPartner = ike->partner;
		ike->partner = partner;
		*partnerP = partner;
	}
	else
		err = kPGPError_OutOfMemory;

done:
	if( IsPGPError(err) && IsntNull( partner ) )
		pgpIKEFreePartner( partner );
	return err;
}

	PGPError
pgpIKEBakeCookie(
	PGPikeContextPriv *		ike,
	PGPUInt32				destAddress,
	PGPUInt32				srcAddress,
	PGPByte *				outCookie )
{
	PGPError				err = kPGPError_NoErr;
	PGPHashContextRef		hash = kInvalidPGPHashContextRef;
	PGPUInt16				port = kPGPike_CommonPort;
	PGPTime					now = PGPGetTime();
	PGPByte					rawOut[32];

	/* cookie recipe per draft-ietf-ipsec-isakmp-10, 2.5.3 */
	if( !ike->cookieDough )
	{
		err = PGPContextGetRandomBytes( ike->pgpContext, &ike->cookieSecret,
									kPGPike_CookieSize ); CKERR;
		ike->cookieDough = TRUE;
	}
	err = PGPNewHashContext( ike->pgpContext, kPGPHashAlgorithm_SHA, &hash); CKERR;
	
	(void)PGPContinueHash( hash, &srcAddress, sizeof(srcAddress) );
	(void)PGPContinueHash( hash, &destAddress, sizeof(destAddress) );
	(void)PGPContinueHash( hash, &port, sizeof(port) );	/* srcPort */
	(void)PGPContinueHash( hash, &port, sizeof(port) );	/* destPort */
	(void)PGPContinueHash( hash, &ike->cookieSecret, sizeof(ike->cookieSecret) );
	(void)PGPContinueHash( hash, &now, sizeof(now) );

	(void)PGPFinalizeHash( hash, rawOut );
	
	pgpCopyMemory( rawOut, outCookie, kPGPike_CookieSize );

done:
	if( PGPHashContextRefIsValid( hash ) )
		PGPFreeHashContext( hash );
	return err;
}

/* The Mighty ISAKMP Packet Interpreter */
	PGPError
pgpIKEHandlePacket(
	PGPikeContextPriv *		ike,
	PGPikeMTPacket *		packet )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner;
	PGPikeExchange *		exchange;
	PGPByte *				mp;
	PGPByte *				ep;
	PGPSize					mLen;
	PGPBoolean				valid = FALSE;
	PGPBoolean				startP1 = FALSE;
	PGPikeCookie			initCookie,
							respCookie;
	PGPikePayload			firstPayload;
	PGPikeExchangeT			exchangeT;
	PGPikeMessageID			messageID;
	PGPUInt8				flags;
	PGPUInt32				thisCksm = 0;
	PGPByte					noMessageID[4] = { 0, 0, 0, 0 };
	PGPByte					noCookie[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
	
	err = pgpIKEHeaderSniffer( ike, packet, &valid );
	if( !valid )
		goto done;
	mp		= packet->packet;
	mLen	= packet->packetSize;
	ep		= mp + mLen;
	
	pgpCopyMemory( mp, initCookie, sizeof( PGPikeCookie ) );
	mp += sizeof( PGPikeCookie );
	pgpCopyMemory( mp, respCookie, sizeof( PGPikeCookie ) );
	mp += sizeof( PGPikeCookie );
	firstPayload	= (PGPikePayload) *mp++;	mp++;
	exchangeT		= (PGPikeExchangeT) *mp++;
	flags			= *mp++;
	pgpCopyMemory( mp, messageID, sizeof( PGPikeMessageID ) );
	mp += sizeof( PGPikeMessageID );
	mp += sizeof( PGPUInt32 );		/* length checked by pgpIKEHeaderSniffer */

	if( ( ( exchangeT == kPGPike_EX_Identity ) || ( exchangeT == kPGPike_EX_Aggressive ) ) &&
		( firstPayload == kPGPike_PY_SA ) &&
		pgpMemoryEqual( &messageID, &noMessageID, sizeof(PGPikeMessageID) ) )
		startP1 = TRUE;
	if( startP1 && ( ike->numExchanges >= kPGPike_MaxExchanges ) )
	{
		err = pgpIKELocalAlert( ike, packet->ipAddress,
				kPGPike_AL_None, kPGPike_IA_TooManyExchanges, FALSE );
		goto done;
	}
	
	for( partner = ike->partner; IsntNull( partner ); partner = partner->nextPartner )
		if( ( partner->ipAddress == packet->ipAddress ) &&
			pgpMemoryEqual( initCookie, partner->initCookie, sizeof(PGPikeCookie) ) &&
			pgpMemoryEqual( respCookie, partner->respCookie, sizeof(PGPikeCookie) ) )
			break;
	
	if( IsNull( partner ) )
	{
		for( partner = ike->partner; IsntNull( partner ); partner = partner->nextPartner )
			if( ( partner->ipAddress == packet->ipAddress ) &&
				( ( exchangeT == kPGPike_EX_Informational ) ||
				pgpMemoryEqual( initCookie, partner->initCookie, sizeof(PGPikeCookie) ) ) &&
				( startP1 || ( exchangeT == kPGPike_EX_Informational ) ||
				pgpMemoryEqual( respCookie, partner->respCookie, sizeof(PGPikeCookie) )) )
				break;
	}
	
#if PGPIKE_DEBUG
	pgpIKEDebugLog( ike, TRUE, "Rcvd: %sexchange=%s, firstPayload=%s%s%s, port=%u\n",
		( flags & kPGPike_ISAKMPEncryptBit ) ? "(E):" : "",
		pgpIKEExchangeTypeString(exchangeT),
		pgpIKEPayloadTypeString(firstPayload),
		( flags & kPGPike_ISAKMPCommitBit ) ? ", Commit" : "",
		( flags & kPGPike_ISAKMPAuthOnlyBit ) ? ", AuthOnly" : "",
		(PGPUInt32)packet->port );
#endif
	if( IsNull( partner ) )
	{
		if( startP1 )
		{
			PGPikeMTSASetup		msg;
			
			if( !pgpMemoryEqual( respCookie, noCookie, sizeof(PGPikeCookie) ) )
			{
#if PGPIKE_DEBUG
				{
					char	ip1Str[20];
					
					pgpIKEGetIPString( packet->ipAddress, ip1Str );
					pgpIKEDebugLog( ike, TRUE, "\tALERT(R): Found ancient cookie from %s (discarded)\n", ip1Str );
				}
#endif
				goto done;
			}
			pgpClearMemory( &msg, sizeof(PGPikeMTSASetup) );
			msg.ipAddress = packet->ipAddress;
			err = (ike->msgProc)( (PGPikeContextRef)ike, ike->userData,
						kPGPike_MT_PolicyCheck, &msg );CKERR;
			if( msg.approved )
			{
				if( ( ( exchangeT == kPGPike_EX_Aggressive ) && !msg.aggressive ) ||
					( !( exchangeT == kPGPike_EX_Aggressive ) && msg.aggressive ) )
				{
#if PGPIKE_DEBUG
					pgpIKEDebugLog( ike, TRUE, "Aggressive mode mismatch\n" );
#endif
					err = pgpIKELocalAlert( ike, packet->ipAddress,
							kPGPike_AL_InvalidExchange, kPGPike_IA_None, FALSE );
					goto done;
				}
				err = pgpIKECreatePartner( ike, &msg, FALSE, &partner );CKERR;
				err = pgpIKECreateExchange( partner, exchangeT, NULL,
									( exchangeT == kPGPike_EX_Aggressive ) ?
									kPGPike_S_AM_WaitSA : kPGPike_S_MM_WaitSA, FALSE,
									&exchange );CKERR;
				pgpCopyMemory( initCookie, partner->initCookie, kPGPike_CookieSize );
				partner->destPort = packet->port;
				if(( packet->port != kPGPike_CommonPort ) &&
                     !IsNATTraversalVendorIDStringPresent(packet->packet, packet->packetSize))
				{
					pgpIKEAbortExchange( &exchange, kPGPike_AL_NATTranslationFailure );
					partner = NULL;
				}
			}
		}
		else
		{
			err = pgpIKELocalAlert( ike, packet->ipAddress,
					kPGPike_AL_InvalidExchange, kPGPike_IA_None, FALSE );
			goto done;
		}
	}
	else
	{
		if( startP1 )
		{
			if( !partner->ready && partner->initiator &&
				IsntNull( partner->exchanges ) )
			{
				if( ( ( partner->exchanges->exchangeT == kPGPike_EX_Aggressive ) &&
						( partner->exchanges->state == kPGPike_S_AM_WaitSA ) ) ||
					( ( partner->exchanges->exchangeT == kPGPike_EX_Identity ) &&
						( partner->exchanges->state == kPGPike_S_MM_WaitSA ) ) )
					pgpCopyMemory( respCookie, partner->respCookie, kPGPike_CookieSize );
				else
				{
#if PGPIKE_DEBUG
					pgpIKEDebugLog( ike, TRUE, "\tDetected repeat packet(1)\n" );
#endif
					partner->rttMillisec *= 2;
					if( partner->rttMillisec > kPGPike_MaxRoundTrip )
						partner->rttMillisec = kPGPike_MaxRoundTrip;
					goto done;
				}
			}
			else
			{
				err = pgpIKELocalAlert( ike, partner->ipAddress,
						kPGPike_AL_InvalidCookie, kPGPike_IA_None, FALSE );
				goto done;
			}
		}
		else if( !pgpMemoryEqual( respCookie, partner->respCookie,
					kPGPike_CookieSize ) )
		{
			/* Force informational exchanges to be encrypted if the
				Phase 1 SA has been setup. */
			if( ( exchangeT == kPGPike_EX_Informational ) && !partner->ready )
			{
				/* super lame versions of IKE send notifications about Phase 1
					negotiations without any of the cookies */
				/*pgpCopyMemory( respCookie, partner->respCookie, kPGPike_CookieSize );*/
			}
			else
			{
				err = pgpIKELocalAlert( ike, partner->ipAddress,
						kPGPike_AL_InvalidCookie, kPGPike_IA_None, FALSE );
				goto done;
			}
		}
	}
	if( IsntNull( partner ) )
	{
		PGPBoolean				found = FALSE;
		
		/* find the exchange */
		for( exchange = partner->exchanges; IsntNull( exchange );
				exchange = exchange->nextExchange )
		{
			if( ( exchange->exchangeT == exchangeT ) &&
				pgpMemoryEqual( &exchange->messageID, messageID,
					sizeof( PGPikeMessageID ) ) )
			{
				found = TRUE;
				break;
			}
		}
		if( !found )
		{
			if( startP1 )
			{
				err = pgpIKECreateExchange( partner, exchangeT, NULL,
						( exchangeT == kPGPike_EX_Aggressive ) ?
						kPGPike_S_AM_WaitSA : kPGPike_S_MM_WaitSA, FALSE,
						&exchange );CKERR;
			}
			else if( ( exchangeT == kPGPike_EX_Informational ) ||
					( ( ( exchangeT == kPGPike_EX_IPSEC_Quick ) ||
						( exchangeT == kPGPike_EX_Transaction ) ) &&
						partner->ready ) )
			{
				if( exchangeT == kPGPike_EX_IPSEC_Quick )
				{
					PGPUInt32	exchIndex;
					
					for( exchIndex = 0; exchIndex < kPGPike_MaxOldExchanges; exchIndex++ )
					{
						if( pgpMemoryEqual( &partner->oldMessageID[exchIndex], messageID, sizeof(PGPikeMessageID) ) )
						{
#if PGPIKE_DEBUG
							pgpIKEDebugLog( ike, TRUE, "\tRemote tried to recycle MessageID, protocol violation\n" );
#endif
							goto done;
						}
					}
				}
				err = pgpIKECreateExchange( partner, exchangeT, &messageID,
						( exchangeT == kPGPike_EX_IPSEC_Quick ) ?
						kPGPike_S_QM_WaitSA : kPGPike_S_ND_OneState,
						FALSE, &exchange );	CKERR;
				if( exchangeT == kPGPike_EX_IPSEC_Quick )
				{
					exchange->destination = (PGPikePendingDest *)PGPNewData( ike->memMgr,
									sizeof(PGPikePendingDest), kPGPMemoryMgrFlags_Clear );
					if( IsNull( exchange->destination ) )
					{
						err = kPGPError_OutOfMemory;
						goto done;
					}
					exchange->destination->ipAddrStart	= 0;
					exchange->destination->ipMaskEnd	= 0;
					exchange->destination->destIsRange	= FALSE;
					exchange->destination->ipPort		= 0;
					exchange->destination->ipProtocol	= 0;
				}
			}
			else
			{
				err = pgpIKELocalAlert( ike,  partner->ipAddress,
							kPGPike_AL_InvalidMessageID, kPGPike_IA_None, FALSE );
				goto done;
			}
		}
		pgpAssertAddrValid( exchange, PGPikeExchange * );
		
		{
			PGPByte *	cp,
					*	ecp;
			
			cp = packet->packet;
			ecp = cp + packet->packetSize;
			while( cp < ecp )
				thisCksm += *cp++;
		}
		/*	Check for repeat packets because this silly protocol
			doesn't have sequence numbers						*/
		if( ( packet->packetSize == exchange->lastRcvdLength ) &&
			( thisCksm == exchange->lastRcvdCksm ) )
		{
			/* just let ourselves time out because we may have already resent */
			partner->rttMillisec *= 2;
			if( partner->rttMillisec > kPGPike_MaxRoundTrip )
				partner->rttMillisec = kPGPike_MaxRoundTrip;
#if PGPIKE_DEBUG
			pgpIKEDebugLog( ike, TRUE, "\tDetected repeat packet(2)\n" );
#endif
			goto done;
		}
		exchange->lastRcvdLength	= packet->packetSize;
		exchange->lastRcvdCksm		= thisCksm;
		
		/* average in estimate of round trip time */
		if( exchange->lastTransmit )
		{
			if( partner->rttMillisec )
			{
				partner->rttMillisec += PGPGetMilliseconds() - exchange->lastTransmit;
				partner->rttMillisec /= 2;
			}
			else
			{
				partner->rttMillisec = PGPGetMilliseconds() - exchange->lastTransmit;
			}
			if( partner->rttMillisec > kPGPike_MaxRoundTrip )
				partner->rttMillisec = kPGPike_MaxRoundTrip;
		}

		if( flags & kPGPike_ISAKMPEncryptBit )
		{
			if( PGPCBCContextRefIsValid( partner->cbc ) )
			{
#if PGPIKE_VERBOSE
				pgpIKEDebugData( partner->ike, "iv", exchange->lastCBC, kPGPike_MaxCipherBlockSize );
#endif
				err = PGPInitCBC( partner->cbc, partner->cipherKey,
									exchange->lastCBC );	CKERR;
				/* Save the IV for the next packet */
				pgpCopyMemory( ep - kPGPike_MaxCipherBlockSize,
								exchange->lastCBC, kPGPike_MaxCipherBlockSize );
				if( ( exchange->exchangeT == kPGPike_EX_Identity ) ||
					( exchange->exchangeT == kPGPike_EX_Aggressive ) )
					pgpCopyMemory( exchange->lastCBC,
								partner->lastP1CBC, kPGPike_MaxCipherBlockSize );
				/* Decrypt packet */
				err = PGPCBCDecrypt( partner->cbc,
								packet->packet + kPGPike_ISAKMPHeaderSize,
								mLen - kPGPike_ISAKMPHeaderSize,
								packet->packet + kPGPike_ISAKMPHeaderSize );CKERR;
			}
			else
			{
				err = pgpIKEAbortExchange( &exchange, 
						kPGPike_AL_InvalidPayload );CKERR;
			}
		}
		if( flags & kPGPike_ISAKMPCommitBit )
		{
			exchange->needsCommit = TRUE;
		}
		if( flags & kPGPike_ISAKMPAuthOnlyBit )
		{
#if PGPIKE_DEBUG
			pgpIKEDebugLog( ike, TRUE, "\tDetected auth-only IKE, rejecting\n" );
#endif
			goto done;
		}
		err = pgpIKEPayloadLengthCheck( partner, firstPayload, mp, ep, &valid );	CKERR;
		if( !valid )
		{
			if( exchange->exchangeT != kPGPike_EX_Informational )
			{
				err = pgpIKEAbortExchange( &exchange,
							kPGPike_AL_UnequalPayloadLengths );	CKERR;
			}
			else
			{
#if PGPIKE_DEBUG
				pgpIKEDebugLog( ike, TRUE, "\tNotify decryption failed (discarded)\n" );
#endif
				err = pgpIKELocalAlert( ike, partner->ipAddress,
						kPGPike_AL_UnequalPayloadLengths, kPGPike_IA_None, FALSE );
				goto done;
			}
		}
		
		/* OK, everything looks cool, now on to the payloads */
		err = pgpIKEProcessPayloads( exchange, firstPayload, mp );CKERR;
	}
	
done:
	return err;	
}

	PGPError
pgpIKEProcessPayloads(
	PGPikeExchange *		exchange,
	PGPikePayload			firstPayload,
	PGPByte *				mp )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner = exchange->partner,
				  *			oPartner;
	PGPikePayload			payload,
							nextPayload;
	PGPUInt16				payLength;
	PGPikeProposal *		proposal = NULL;
	PGPipsecSA *			sa;
	PGPBoolean				rcvdSA = FALSE;
	PGPBoolean				rcvdHash = FALSE;
	PGPBoolean				found = FALSE;
	PGPBoolean				sendRL = FALSE;
	PGPByte					hashPY[kPGPike_MaxHashSize];
	PGPHMACContextRef		p2hmac = kInvalidPGPHMACContextRef;
	PGPikeNDPayload	*		ndPays = NULL,
					*		wNDPays;
	PGPBoolean				sharedKey;
	PGPikeMTAuthCheck *		authCheck = NULL;

	if( ( ( ( exchange->state == kPGPike_S_MM_WaitSA ) ||
			( exchange->state == kPGPike_S_AM_WaitSA ) ) &&
			( firstPayload != kPGPike_PY_SA ) ) ||
		( ( exchange->state == kPGPike_S_QM_WaitSA ) &&
			( ( firstPayload != kPGPike_PY_Hash ) ||
			  ( (PGPikePayload)*mp != kPGPike_PY_SA ) ) ) )
	{
		err = pgpIKEAbortExchange( &exchange,
						kPGPike_AL_InvalidPayload );
		goto done;
	}
	if( ( exchange->state == kPGPike_S_ND_OneState ) &&
		  	partner->ready && ( firstPayload != kPGPike_PY_Hash ) )
	{
#if PGPIKE_DEBUG
		pgpIKEDebugLog( partner->ike, TRUE, "\tRcvd unencrypted Notify linked to P1 (discarded)\n" );
#endif
		err = pgpIKELocalAlert( exchange->ike, partner->ipAddress,
				kPGPike_AL_PayloadMalformed, kPGPike_IA_None, FALSE );
		goto done;
	}
	if( PGPCBCContextRefIsValid( partner->cbc ) &&
		( ( exchange->exchangeT == kPGPike_EX_IPSEC_Quick ) ||
		( ( ( exchange->exchangeT == kPGPike_EX_Informational ) ||
			( exchange->exchangeT == kPGPike_EX_Transaction ) ) &&
			( firstPayload == kPGPike_PY_Hash ) ) ) )
	{
		pgpClearMemory( hashPY, kPGPike_MaxHashSize );
		err = PGPNewHMACContext( partner->ike->pgpContext,
						partner->sdkHashAlg,
						partner->skeyid_a,
						partner->hashSize,
						&p2hmac );	CKERR;
		if( !exchange->initiator && ( exchange->state == kPGPike_S_QM_WaitHash3 ) )
		{
			err = PGPContinueHMAC( p2hmac, "\0", 1 );	CKERR;
		}
		err = PGPContinueHMAC( p2hmac, exchange->messageID,
								sizeof(PGPikeMessageID) );	CKERR;
		if( exchange->exchangeT == kPGPike_EX_IPSEC_Quick )
		{
			if( ( exchange->initiator && ( exchange->state == kPGPike_S_QM_WaitSA ) ) ||
				( !exchange->initiator && ( exchange->state == kPGPike_S_QM_WaitHash3 ) ) )
				err = PGPContinueHMAC( p2hmac, exchange->initNonce,
											exchange->initNonceLen );CKERR;
			if( !exchange->initiator && ( exchange->state == kPGPike_S_QM_WaitHash3 ) )
				err = PGPContinueHMAC( p2hmac, exchange->respNonce,
											exchange->respNonceLen );CKERR;
		}
	}
	payload = firstPayload;
	while( payload != kPGPike_PY_None )
	{
		PGPByte	*	pp,
				*	ep;
		
		nextPayload = (PGPikePayload)*mp++;
		mp++;	/* reserved */
		payLength = PGPEndianToUInt16( kPGPBigEndian, mp );
		mp += sizeof(PGPUInt16);
		if( PGPHMACContextRefIsValid( p2hmac ) && ( payload != kPGPike_PY_Hash ) )
		{
			err = PGPContinueHMAC( p2hmac, mp - kPGPike_ISAKMPPayloadHeaderSize,
							payLength );	CKERR;
		}
		payLength -= kPGPike_ISAKMPPayloadHeaderSize;
		pp = mp;
		ep = mp + payLength;
		switch( payload )
		{
			case kPGPike_PY_SA:
			{
				PGPikeDOI			doi;
				PGPUInt32			situation;
				PGPikePayload		nextSAPayload;
				PGPUInt16			saPayLength;
				PGPikeAlert			alert = kPGPike_AL_None;
				PGPikeProposal *	nproposal;
				PGPBoolean			valid = FALSE;
				
				if( rcvdSA )	/* we only handle one SA payload */
					break;
				rcvdSA = TRUE;
				if( ( exchange->state != kPGPike_S_QM_WaitSA ) &&
					( exchange->state != kPGPike_S_MM_WaitSA ) &&
					( exchange->state != kPGPike_S_AM_WaitSA ) )
					goto invalidPayload;
				if( !exchange->initiator &&
					( ( exchange->state == kPGPike_S_MM_WaitSA ) ||
					  ( exchange->state == kPGPike_S_AM_WaitSA ) ) )
				{
					/* save the SA payload body for HASH_X */
					pgpAssert( IsNull( exchange->initSABody ) );
					exchange->initSABodySize = payLength;
					exchange->initSABody = PGPNewData(
									exchange->ike->memMgr,
									payLength, kPGPMemoryMgrFlags_Clear );
					if( IsNull( exchange->initSABody ) )
					{
						err = kPGPError_OutOfMemory;
						goto done;
					}
					pgpCopyMemory( pp, exchange->initSABody, payLength );
				}
				if( ep - pp < sizeof(PGPUInt32) * 2 )
				{
					err = pgpIKEAbortExchange( &exchange,
								kPGPike_AL_UnequalPayloadLengths );
					goto done;
				}
				/* doi */
				doi = (PGPikeDOI)PGPEndianToUInt32( kPGPBigEndian, pp );
				pp += sizeof( PGPikeDOI );
				if( doi != kPGPike_DOI_IPSEC )
				{
					err = pgpIKEAbortExchange( &exchange,
								kPGPike_AL_DOIUnsupported );
					goto done;
				}
				/* situation */
				situation = PGPEndianToUInt32( kPGPBigEndian, pp );
				pp += sizeof( PGPUInt32 );
				if( ( situation & kPGPike_IPSEC_SIT_Integrity ) ||
					( situation & kPGPike_IPSEC_SIT_Secrecy ) ||
					!( situation & kPGPike_IPSEC_SIT_IdentityOnly ))
				{
					err = pgpIKEAbortExchange( &exchange,
								kPGPike_AL_SituationUnsupported );
					goto done;
				}
				/* Proposals */
				do
				{
					nproposal = NULL;
					
					if( ep - pp < kPGPike_ISAKMPPayloadHeaderSize )
					{
						err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_UnequalPayloadLengths );
						goto done;
					}
					nextSAPayload = (PGPikePayload)*pp++;	pp++;
					saPayLength = PGPEndianToUInt16( kPGPBigEndian, pp ) -
									kPGPike_ISAKMPPayloadHeaderSize;
					pp += sizeof(PGPUInt16);
					
					err = pgpIKEProcessProposal( &exchange, pp, ep,
												&alert, &nproposal );
					if( IsPGPError( err ) || ( alert != kPGPike_AL_None ) )
						goto done;
					if( IsntNull( nproposal ) )
					{
						PGPikeProposal *wproposal = proposal;
						
						if( IsNull( wproposal ) )
							proposal = nproposal;
						else
						{
							while( IsntNull( wproposal->nextProposal ) )
								wproposal = wproposal->nextProposal;
							wproposal->nextProposal = nproposal;
						}
					}
					pp += saPayLength;
				} while( nextSAPayload == kPGPike_PY_Proposal );
				if( IsNull( proposal ) ||
					( ( exchange->state == kPGPike_S_AM_WaitSA ) &&
					  ( proposal->numTransforms > 1 ) ) )
				{
					if( alert == kPGPike_AL_None )
						err = pgpIKEAbortExchange( &exchange, kPGPike_AL_BadProposal );
					goto done;
				}
				err = pgpIKESelectProposal( exchange, proposal, &valid );	CKERR;
				if( !valid )
				{
					err = pgpIKEAbortExchange( &exchange, kPGPike_AL_NoProposalChoice );
					goto done;
				}
				else if( ( exchange->exchangeT == kPGPike_EX_Aggressive ) &&
						 !exchange->initiator )
				{
					err = pgpIKELoadGroup( exchange );	CKERR;
				}
				break;
			}
			case kPGPike_PY_KeyExchange:
				if( ( exchange->state == kPGPike_S_MM_WaitKE ) ||
					( exchange->state == kPGPike_S_AM_WaitSA ) ||
					( exchange->state == kPGPike_S_QM_WaitSA ) )
				{
					err = pgpIKEProcessKE( &exchange, pp, payLength );	CKERR;
				}
				else
					goto invalidPayload;
				break;
			case kPGPike_PY_Identification:
				if( exchange->exchangeT == kPGPike_EX_IPSEC_Quick )
				{
					if( exchange->initiator )
					{
						PGPBoolean	goodID = FALSE;
						
						if( !exchange->checkedIDci )
						{
							if( ( payLength == exchange->idBodySize ) &&
								pgpMemoryEqual( pp, exchange->idBody, payLength ) )
							{
								exchange->checkedIDci = TRUE;
								goodID = TRUE;
							}
						}
						else if( !exchange->checkedIDcr )
						{
							if( ( payLength == exchange->idRBodySize ) &&
								pgpMemoryEqual( pp, exchange->idRBody, payLength ) )
							{
								exchange->checkedIDcr = TRUE;
								goodID = TRUE;
							}
						}
						if( !goodID )
						{
#if PGPIKE_DEBUG
							pgpIKEDebugLog( exchange->ike, TRUE,
								"\tReceived IDs do not match sent IDs\n" );
#endif
							err = pgpIKEAbortExchange( &exchange, kPGPike_AL_InvalidID );
							goto done;
						}
					}
					else
					{
						if( IsNull( exchange->idBody ) )
						{
							/* save the ID payload for HASH_X */
							if( IsntNull( exchange->idBody ) )
							{
								(void)PGPFreeData( exchange->idBody );
								exchange->idBody = NULL;
							}
							exchange->idBodySize = payLength;
							exchange->idBody = PGPNewData( exchange->ike->memMgr,
													payLength, kPGPMemoryMgrFlags_Clear );
							if( IsNull( exchange->idBody ) )
							{
								err = kPGPError_OutOfMemory;
								goto done;
							}
							pgpCopyMemory( pp, exchange->idBody, payLength );
		#if PGPIKE_DEBUG
							pgpIKEDebugLog( exchange->ike, TRUE,
								"\tRemote ID(P2): %s\n", pgpIKEIDTypeString( (PGPipsecIdentity)*pp ) );
		#endif
							pp += sizeof(PGPUInt32);
						}
						else if( IsNull( exchange->idRBody ) )
						{
							exchange->idRBodySize = payLength;
							exchange->idRBody = PGPNewData( exchange->ike->memMgr,
								payLength, kPGPMemoryMgrFlags_Clear );
							if( IsNull( exchange->idRBody ) )
							{
								err = kPGPError_OutOfMemory;
								goto done;
							}
							pgpCopyMemory( pp, exchange->idRBody, payLength );
						}
						else
						{
#if PGPIKE_DEBUG
							pgpIKEDebugLog( exchange->ike, TRUE, "\tToo many IDs rcvd.\n" );
#endif
							err = pgpIKEAbortExchange( &exchange, kPGPike_AL_InvalidID );
							goto done;
						}
					}
				}
				else if( ( exchange->exchangeT == kPGPike_EX_Identity ) ||
						 ( exchange->state == kPGPike_S_AM_WaitSA ) )
				{
					PGPByte		ipProtocol;
					PGPUInt16	ipPort;
					
					/* save the ID payload for HASH_X */
					if( exchange->initiator )
					{
						if( IsntNull( exchange->idRBody ) )
						{
							(void)PGPFreeData( exchange->idRBody );
							exchange->idRBody = NULL;
						}
						exchange->idRBodySize = payLength;
						exchange->idRBody = PGPNewData( exchange->ike->memMgr,
												payLength, kPGPMemoryMgrFlags_Clear );
						if( IsNull( exchange->idRBody ) )
						{
							err = kPGPError_OutOfMemory;
							goto done;
						}
						pgpCopyMemory( pp, exchange->idRBody, payLength );
					}
					else
					{
						if( IsntNull( exchange->idBody ) )
						{
							(void)PGPFreeData( exchange->idBody );
							exchange->idBody = NULL;
						}
						exchange->idBodySize = payLength;
						exchange->idBody = PGPNewData( exchange->ike->memMgr,
												payLength, kPGPMemoryMgrFlags_Clear );
						if( IsNull( exchange->idBody ) )
						{
							err = kPGPError_OutOfMemory;
							goto done;
						}
						pgpCopyMemory( pp, exchange->idBody, payLength );
					}
#if PGPIKE_DEBUG
					pgpIKEDebugLog( exchange->ike, TRUE,
						"\tRemote ID(P1): %s\n", pgpIKEIDTypeString( (PGPipsecIdentity)*pp ) );
#endif
					pp++;
					ipProtocol = *pp++;
					ipPort = PGPEndianToUInt16( kPGPBigEndian, pp );
					pp += sizeof(PGPUInt16);
					if( ( ( ( ipProtocol != 0 ) && (ipProtocol != kPGPike_IPProtocolUDP ) ) ||
						( ( ipPort != 0 ) && ( ipPort != kPGPike_CommonPort ) ) ) )
					{
						err = pgpIKEAbortExchange( &exchange, kPGPike_AL_InvalidID );
						goto done;
					}
				}
				else
					goto invalidPayload;
				break;
			case kPGPike_PY_Certificate:
				if( ( ( exchange->state == kPGPike_S_MM_WaitFinal ) ||
					  ( ( exchange->exchangeT == kPGPike_EX_Aggressive ) &&
					    ( ( !exchange->initiator && ( exchange->state == kPGPike_S_AM_WaitFinal ) ) ||
					    ( exchange->initiator && ( exchange->state == kPGPike_S_AM_WaitSA ) ) ) ) )  &&
					( ( exchange->proposals->t[0].u.ike.authMethod == kPGPike_AM_DSS_Sig ) ||
						( exchange->proposals->t[0].u.ike.authMethod == kPGPike_AM_RSA_Sig ) ) )
				{
					err = pgpIKEProcessCert( &exchange, pp, payLength );	CKERR;
				}
				else
					goto invalidPayload;
				break;
			case kPGPike_PY_Signature:
				if( ( ( exchange->state == kPGPike_S_MM_WaitFinal ) ||
					  ( ( exchange->exchangeT == kPGPike_EX_Aggressive ) &&
					    ( ( !exchange->initiator && ( exchange->state == kPGPike_S_AM_WaitFinal ) ) ||
					    ( exchange->initiator && ( exchange->state == kPGPike_S_AM_WaitSA ) ) ) ) ) &&
					( ( exchange->proposals->t[0].u.ike.authMethod == kPGPike_AM_DSS_Sig ) ||
						( exchange->proposals->t[0].u.ike.authMethod == kPGPike_AM_RSA_Sig ) ) )
				{
					err = pgpIKEProcessSig( &exchange, pp, payLength );		CKERR;
				}
				else
					goto invalidPayload;
				break;
			case kPGPike_PY_CertRequest:
				if( ( exchange->exchangeT == kPGPike_EX_Identity ) ||
					( ( exchange->exchangeT == kPGPike_EX_Aggressive ) &&
					  ( exchange->state == kPGPike_S_AM_WaitSA ) ) )
				{
					err = pgpIKEProcessCertRequest( &exchange, pp, payLength );CKERR;
				}
				else
					goto invalidPayload;
				break;
			case kPGPike_PY_Hash:
				if( !rcvdHash && ( partner->hashSize == payLength ) )
				{
					switch( exchange->state )
					{
						case kPGPike_S_AM_WaitSA:
							if( !exchange->initiator )
								goto invalidPayload;
						case kPGPike_S_AM_WaitFinal:
						case kPGPike_S_MM_WaitFinal:
							switch( exchange->proposals->t[0].u.ike.authMethod )
							{
								case kPGPike_AM_PreSharedKey:
								case kPGPike_AM_RSA_Encrypt:
								case kPGPike_AM_RSA_Encrypt_R:
								{
									PGPByte		lHash[kPGPike_MaxHashSize];
									
									err = pgpIKEGetAuthHash( exchange,
												(PGPBoolean)!exchange->initiator,
												lHash );	CKERR;
									if( pgpMemoryEqual( lHash, pp, partner->hashSize ) )
									{
										partner->authenticated = TRUE;
#if PGPIKE_DEBUG
										pgpIKEDebugLog( exchange->ike, TRUE, "\tAuthenticated Hash\n" );
#endif
									}
									else
									{
#if PGPIKE_DEBUG
										pgpIKEDebugLog( exchange->ike, TRUE, "\tHash Authentication failed\n" );
#endif
										err = pgpIKEAbortExchange( &exchange,
													kPGPike_AL_AuthenticationFailed );
										goto done;
									}
									break;
								}
								default:
									goto invalidPayload;
							}
							break;
						case kPGPike_S_QM_WaitSA:
						case kPGPike_S_QM_WaitHash3:
						case kPGPike_S_ND_OneState:
							pgpCopyMemory( pp, hashPY, payLength );
							break;
						default:
							goto invalidPayload;
					}
					rcvdHash = TRUE;
				}
				else
					goto invalidPayload;	
				break;
			case kPGPike_PY_Nonce:
				if( ( exchange->state == kPGPike_S_MM_WaitKE ) ||
					( exchange->state == kPGPike_S_QM_WaitSA ) ||
					( exchange->state == kPGPike_S_AM_WaitSA ) )
				{
					if( ( payLength >= kPGPike_MinNonceSize ) &&
						( payLength <= kPGPike_MaxNonceSize ) )
					{
						if( exchange->initiator )
						{
							exchange->respNonceLen = payLength;
							pgpCopyMemory( pp, exchange->respNonce,
											payLength );
							if( exchange->state == kPGPike_S_AM_WaitSA )
								err = pgpIKEGoSecure( exchange ); CKERR;
						}
						else
						{
							exchange->initNonceLen = payLength;
							pgpCopyMemory( pp, exchange->initNonce,
											payLength );
						}
					}
					else
						goto invalidPayload;
				}
				else
					goto invalidPayload;
				break;
			case kPGPike_PY_Attribute:
				if( ( exchange->exchangeT == kPGPike_EX_Transaction ) &&
					exchange->partner->ready && ( exchange->initiator ||
					( exchange->partner->authStyle != kPGPike_AS_Normal ) ) )
				{
					err = pgpIKEProcessConfig( &exchange, pp, payLength );CKERR;
				}
				else
					goto invalidPayload;
				break;
			case kPGPike_PY_Notification:
			case kPGPike_PY_Delete:
			{
				PGPikeNDPayload *	newND;
				
				/* save off the notify and delete payloads to make
					sure the QM hash is approved prior to action */
				newND = PGPNewData( exchange->ike->memMgr,
								sizeof(PGPikeNDPayload), kPGPMemoryMgrFlags_Clear );
				if( IsntNull( newND ) )
					newND->pay = PGPNewData( exchange->ike->memMgr,
								payLength, 0 );
				if( IsntNull( newND ) && IsntNull( newND->pay ) )
				{
					pgpCopyMemory( pp, newND->pay, payLength );
					newND->payLen	= payLength;
					newND->payType	= payload;
					newND->nextND	= ndPays;
					ndPays			= newND;
				}
				else
				{
					err = kPGPError_OutOfMemory;
					goto done;
				}
				break;
			}
			case kPGPike_PY_VendorID:
			{
				PGPSize	vidSize = strlen( kPGPike_PGPVendorString1 );
				
#if PGPIKE_VERBOSE
				pgpIKEDebugData( exchange->ike, "VendorID", pp, payLength );
#endif
				if( ( payLength >= vidSize ) &&
					pgpMemoryEqual( kPGPike_PGPVendorString1, pp, vidSize ) )
				{
					partner->remotePGPVersion = 1;
					if( payLength > vidSize )
					{
						partner->remotePGPVersion = 2;
#if PGPIKE_DEBUG
						pgpIKEDebugLog( exchange->ike, TRUE, "\tDetected PGPikeV2\n" );
#endif
					}
					else
					{
#if PGPIKE_DEBUG
						pgpIKEDebugLog( exchange->ike, TRUE, "\tDetected PGPikeV1\n" );
#endif
					}
				}
				else if( ( payLength == sizeof(kPGPike_XAuth6VendorID) ) &&
							pgpMemoryEqual( kPGPike_XAuth6VendorID, pp, payLength ) )
				{
#if PGPIKE_DEBUG
					pgpIKEDebugLog( exchange->ike, TRUE, "\tDetected remote XAuth V6+ support\n" );
#endif
				}
				else if( ( payLength == sizeof(kPGPike_NATTraversalVendorID) ) &&
					        pgpMemoryEqual(kPGPike_NATTraversalVendorID, pp, payLength ) )

				{
#if PGPIKE_DEBUG
					pgpIKEDebugLog( exchange->ike, TRUE, "\tDetected remote NAT Traversal support\n" );
#endif
                    exchange->rcvdNATTravVendorID = TRUE;
				}
				break;
			}
			case kPGPike_PY_NATDiscovery:
				if( ( ep - pp ) == kPGPike_MD5HashSize )
				{
					PGPByte	natHash[kPGPike_MD5HashSize];
					
					err = pgpIKEGetNATDiscoveryHash( exchange, natHash );	CKERR;
					if( pgpMemoryEqual( natHash, pp, kPGPike_MD5HashSize ) )
					{
#if PGPIKE_DEBUG
						if( partner->bNATTraversal )
							pgpIKEDebugLog( exchange->ike, TRUE, "\tNATTraversal is Forced\n");
						else
							pgpIKEDebugLog( exchange->ike, TRUE, "\tNATTraversal not required\n");
#endif
					}
					else
					{
#if PGPIKE_DEBUG
						pgpIKEDebugLog( exchange->ike, TRUE, "\tNATTraversal is Required\n");
#endif
						partner->bNATTraversal = TRUE;
					}
				}
				else
					goto invalidPayload;
				break;
			default:
			invalidPayload:
				err = pgpIKEAbortExchange( &exchange,
								kPGPike_AL_InvalidPayload );
				goto done;
		}
		payload = nextPayload;
		mp += payLength;
	}
	if( PGPHMACContextRefIsValid( p2hmac ) )
	{
		PGPByte *	p2hmacFinal[kPGPike_MaxHashSize];
		
		err = PGPFinalizeHMAC( p2hmac, p2hmacFinal );	CKERR;
		if( !rcvdHash || !pgpMemoryEqual( p2hmacFinal, hashPY, partner->hashSize ) )
		{
			err = pgpIKEAbortExchange( &exchange,
							kPGPike_AL_AuthenticationFailed );
			goto done;
		}
	}
	if( IsntNull( ndPays ) )
	{
		for( wNDPays = ndPays; IsntNull( wNDPays); wNDPays = wNDPays->nextND )
		{
			err = pgpIKEProcessInformational( exchange->ike,
					exchange->partner->ipAddress, &exchange, wNDPays );	CKERR;
			if( IsNull( exchange ) )
				goto done;
			
		}
	}
	if( exchange->initiator )
	{
		switch( exchange->state )
		{
			case kPGPike_S_MM_WaitSA:
				switch( exchange->proposals->t[0].u.ike.authMethod )
				{
					case kPGPike_AM_PreSharedKey:
					case kPGPike_AM_DSS_Sig:
					case kPGPike_AM_RSA_Sig:
						err = pgpIKELoadGroup( exchange );	CKERR;
						err = pgpIKEDoPacket( &exchange,
								kPGPike_PY_KeyExchange,
								kPGPike_PY_Nonce,
								exchange->rcvdNATTravVendorID ? kPGPike_PY_NATDiscovery : kPGPike_PY_Skip,
								kPGPike_PY_None );	CKERR;
						break;
					case kPGPike_AM_RSA_Encrypt:
					case kPGPike_AM_RSA_Encrypt_R:
					default:
						pgpAssert( 0 );	/* unsupported */
						break;
				}
				exchange->state = kPGPike_S_MM_WaitKE;
				break;
			case kPGPike_S_MM_WaitKE:
				found = FALSE;
				if( ( exchange->respNonceLen == 0 ) || ( IsNull( exchange->dhYr ) ) )
				{
					err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_InvalidPayload );
					goto done;
				}
				/* encrypto mondo */
				err = pgpIKEGoSecure( exchange );	CKERR;
				
				for( oPartner = exchange->ike->partner; IsntNull( oPartner );
						oPartner = oPartner->nextPartner )
				{
					if( ( oPartner->ipAddress == partner->ipAddress ) &&
						( oPartner != partner ) )
						found = TRUE;
				}
				exchange->outAlert = kPGPike_AL_InitialContact;
				exchange->outInfoProtocol = kPGPike_PR_IKE;
				exchange->outInfoSPICount = 0;

				if( partner->authStyle == kPGPike_AS_HybridAuth )
				{
					switch( exchange->proposals->t[0].u.ike.authMethod )
					{
						case kPGPike_AM_DSS_Sig:
						case kPGPike_AM_RSA_Sig:
							err = pgpIKEDoPacket( &exchange,
									kPGPike_PY_Identification,
									kPGPike_PY_Hash,
									!found ? kPGPike_PY_Notification : kPGPike_PY_Skip,
									kPGPike_PY_None );	CKERR;
							break;
						default:
							pgpAssert( 0 );	/* unsupported */
							break;
					}
				}
				else switch( exchange->proposals->t[0].u.ike.authMethod )
				{
					case kPGPike_AM_RSA_Encrypt:
					case kPGPike_AM_RSA_Encrypt_R:
						err = pgpIKEDoPacket( &exchange,
								kPGPike_PY_Hash,
								!found ? kPGPike_PY_Notification : kPGPike_PY_Skip,
								kPGPike_PY_None );	CKERR;
						break;
					case kPGPike_AM_PreSharedKey:
						err = pgpIKEDoPacket( &exchange,
								kPGPike_PY_Identification,
								kPGPike_PY_Hash,
								!found ? kPGPike_PY_Notification : kPGPike_PY_Skip,
								kPGPike_PY_None );	CKERR;
						break;
					case kPGPike_AM_DSS_Sig:
					case kPGPike_AM_RSA_Sig:
						err = pgpIKEDoPacket( &exchange,
								kPGPike_PY_Identification,
								kPGPike_PY_Certificate,
								kPGPike_PY_Signature,
								kPGPike_PY_CertRequest,
								!found ? kPGPike_PY_Notification : kPGPike_PY_Skip,
								kPGPike_PY_None );	CKERR;
						break;
					default:
						pgpAssert( 0 );	/* unsupported */
						break;
				}
				exchange->state = kPGPike_S_MM_WaitFinal;
				break;
			case kPGPike_S_AM_WaitSA:
				if( ( exchange->respNonceLen == 0 ) || ( IsNull( exchange->dhYr ) ) )
				{
					err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_InvalidPayload );
					goto done;
				}
				found = FALSE;
				for( oPartner = exchange->ike->partner; IsntNull( oPartner );
						oPartner = oPartner->nextPartner )
				{
					if( ( oPartner->ipAddress == partner->ipAddress ) &&
						( oPartner != partner ) )
						found = TRUE;
				}
				exchange->outAlert = kPGPike_AL_InitialContact;
				exchange->outInfoProtocol = kPGPike_PR_IKE;
				exchange->outInfoSPICount = 0;
				switch( exchange->proposals->t[0].u.ike.authMethod )
				{
					case kPGPike_AM_PreSharedKey:
						err = pgpIKEDoPacket( &exchange,
								kPGPike_PY_Hash,
								!found ? kPGPike_PY_Notification : kPGPike_PY_Skip,
								exchange->rcvdNATTravVendorID ? kPGPike_PY_NATDiscovery : kPGPike_PY_Skip,
								kPGPike_PY_None );	CKERR;
						break;
					case kPGPike_AM_DSS_Sig:
					case kPGPike_AM_RSA_Sig:
						err = pgpIKEDoPacket( &exchange,
								kPGPike_PY_Certificate,
								kPGPike_PY_Signature,
								!found ? kPGPike_PY_Notification : kPGPike_PY_Skip,
								exchange->rcvdNATTravVendorID ? kPGPike_PY_NATDiscovery : kPGPike_PY_Skip,
								kPGPike_PY_None );	CKERR;
						break;
					default:
						pgpAssert( 0 );	/* unsupported */
						break;
				}
				/* FALLTHROUGH */
			case kPGPike_S_MM_WaitFinal:
				if( partner->authenticated )
				{
					partner->kbLifeTime		= exchange->proposals->kbLifeTime;
					partner->secLifeTime	= exchange->proposals->secLifeTime;
					partner->birthTime		= PGPGetTime();
					if( IsntNull( partner->pgpAuthKey.pass ) )
						PGPFreeData( partner->pgpAuthKey.pass );
					partner->pgpAuthKey.pass = NULL;
					partner->pgpAuthKey.passLength = 0;
					if( IsntNull( partner->x509AuthKey.pass ) )
						PGPFreeData( partner->x509AuthKey.pass );
					partner->x509AuthKey.pass = NULL;
					partner->x509AuthKey.passLength = 0;
					partner->ready			= TRUE;
					err = pgpIKEFreeExchange( exchange );
					exchange = NULL;
					err = pgpIKELocalAlert( partner->ike, partner->ipAddress,
							kPGPike_AL_None, kPGPike_IA_NewPhase1SA, FALSE );
#if PGPIKE_DEBUG
					pgpIKEDebugLog( partner->ike, TRUE, "\tPhase 1 SA Negotiated(I)\n" );
#endif
					/* Phase 1 complete, begin Phase 2 if necessary */
					if( PGPIKE_XAUTHCISCO || ( partner->authStyle == kPGPike_AS_Normal ) )
					{
						if( partner->virtualIP )
						{
							/* if mode-cfg is requested, we initiate a Request transaction */
							err = pgpIKEStartConfigRequest( partner, &exchange );	CKERR;
						}
						else if( IsntNull( partner->pendList ) )
						{
							err = pgpIKEStartQMExchange( partner, &exchange );	CKERR;
						}
					}
					if( ( partner->authStyle != kPGPike_AS_Normal ) && !partner->termSchedule )
						partner->termSchedule = PGPGetTime() + kPGPike_XAuthSlack;
				}
				else
				{
					err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_AuthenticationFailed );
					goto done;
				}
				break;
			case kPGPike_S_QM_WaitSA:
				if( exchange->respNonceLen > 0 )
				{
					if( IsntNull( exchange->idBody ) &&
						( !exchange->checkedIDci || !exchange->checkedIDcr ) )
					{
#if PGPIKE_DEBUG
						pgpIKEDebugLog( exchange->ike, TRUE, "\tOnly one client ID rcvd!\n" );
#endif
						err = pgpIKEAbortExchange( &exchange,
										kPGPike_AL_InvalidID );
						goto done;
					}
					if( ( exchange->proposals->t[0].u.ipsec.groupID !=
						kPGPike_GR_None ) && IsNull( exchange->dhYr ) )
					{
						err = pgpIKEAbortExchange( &exchange,
										kPGPike_AL_AuthenticationFailed );
						goto done;
					}
					exchange->state = kPGPike_S_QM_WaitHash3;
					err = pgpIKEDoPacket( &exchange, kPGPike_PY_None );CKERR;
					/* Phase 2 complete.  Generate the SAs */
					if( !exchange->needsCommit )
					{
						err = pgpIKECompletePhase2( exchange, &sa );				CKERR;
						err = pgpIKESAEstablished( partner, sa );					CKERR;
						err = pgpIKEFreeExchange( exchange );						CKERR;
						exchange = NULL;
					}
					else
						exchange->complete = TRUE;
				}
				else
				{
					err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_InvalidPayload );
					goto done;
				}
				break;
			case kPGPike_S_QM_WaitHash3:
				/* we would only legally receive something in this state
					when the commit bit was set by the responder and we
					are waiting for a connected notify message */
				if( !exchange->needsCommit )
				{
					err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_InvalidPayload );
					goto done;
				}
				break;
			case kPGPike_S_ND_OneState:
				if( ( exchange->exchangeT == kPGPike_EX_Transaction ) &&
					( exchange->numConfigs > 0 ) )
				{
					PGPUInt16			configIndex;
					
					for( configIndex = 0; configIndex < exchange->numConfigs; configIndex++ )
					{
						if( exchange->configs[configIndex].configType == kPGPike_CT_Reply )
						{
							switch( exchange->configs[configIndex].attribute )
							{
								case kPGPike_AT_Config_IPv4Address:
									partner->assignedIP =
										exchange->configs[configIndex].value;
									break;
								case kPGPike_AT_Config_IPv4Mask:
									partner->assignedMask =
										exchange->configs[configIndex].value;
									break;
								case kPGPike_AT_Config_IPv4DNS:
									partner->assignedDNS =
										exchange->configs[configIndex].value;
									break;
								case kPGPike_AT_Config_IPv4NBNS:
									partner->assignedWINS =
										exchange->configs[configIndex].value;
									break;
								case kPGPike_AT_Config_IPv4DHCP:
									partner->assignedDHCP =
										exchange->configs[configIndex].value;
									break;
								case kPGPike_AT_Config_AddressExpiry:
									if( exchange->configs[configIndex].value >
										kPGPike_SecLifeRekeySlack )
									{
										partner->assignedExpiry =
											PGPGetTime() + exchange->configs[configIndex].value;
									}
									break;
								default:
#if PGPIKE_DEBUG
									pgpIKEDebugLog( partner->ike, TRUE, "\tRemote violation, unknown attribute (%u)\n",
										(PGPUInt32)exchange->configs[configIndex].attribute );
#endif
									break;
							}
						}
						else
						{
							err = pgpIKEAbortExchange( &exchange, kPGPike_AL_UnsupportedExchange );
							goto done;
						}
					}
					found = FALSE;
					for( sa = partner->ike->sa; IsntNull( sa ); sa = sa->nextSA )
						if( ( sa->ipAddress == partner->ipAddress ) && sa->activeOut )
							found = TRUE;
					if( found )
					{
						PGPikeMTIdentity	mtIdentity;
						
						pgpClearMemory( &mtIdentity, sizeof(PGPikeMTIdentity) );
						mtIdentity.active		= TRUE;
						mtIdentity.ipAddress	= partner->ipAddress;
						mtIdentity.assignedIP	= partner->assignedIP;
						mtIdentity.assignedDNS	= partner->assignedDNS;
						mtIdentity.assignedWINS	= partner->assignedWINS;
						err = (partner->ike->msgProc)(
								(PGPikeContextRef)partner->ike, partner->ike->userData,
								kPGPike_MT_Identity, &mtIdentity );	CKERR;
					}
					for( sa = partner->ike->sa; IsntNull( sa ); sa = sa->nextSA )
						if( ( sa->ipAddress == partner->ipAddress ) && sa->activeOut )
						{
							sa->assignedIP		= partner->assignedIP;
							sa->assignedDNS		= partner->assignedDNS;
							sa->assignedWINS	= partner->assignedWINS;
							err = (partner->ike->msgProc)(
									(PGPikeContextRef)partner->ike, partner->ike->userData,
									kPGPike_MT_SAUpdate, sa );	CKERR;
						}
					err = pgpIKEFreeExchange( exchange );		CKERR;
					exchange = NULL;
					
					/* if any other P1 SAs exist for this gateway, make sure
						to zero out the virtual IP so we don't accidentally
						notify the application that it has been released. */
					{
						PGPikePartner *	walkPartner;
						
						for( walkPartner = partner->ike->partner; IsntNull( walkPartner );
								walkPartner = walkPartner->nextPartner )
						{
							if( ( walkPartner != partner ) &&
								( walkPartner->ipAddress == partner->ipAddress ) &&
								walkPartner->ready && walkPartner->assignedIP )
							{
								walkPartner->assignedIP = 0;
							}
						}
					}
				}
				break;
			default:
				break;
		}
	}
	else
	{
		switch( exchange->state )
		{
			case kPGPike_S_AM_WaitSA:
			{
				PGPBoolean	sendXAuthVendor = ( exchange->partner->authStyle != kPGPike_AS_Normal );
				
				if( ( exchange->initNonceLen == 0 ) || ( IsNull( exchange->dhYi ) ) )
				{
					err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_InvalidPayload );
					goto done;
				}
				switch( exchange->proposals->t[0].u.ike.authMethod )
				{
					case kPGPike_AM_PreSharedKey:
					case kPGPike_AM_XAuth_RespPreShared:
						err = pgpIKEDoPacket( &exchange,
									kPGPike_PY_SA,
									kPGPike_PY_KeyExchange,
									kPGPike_PY_Nonce,
									kPGPike_PY_Identification,
									exchange->rcvdNATTravVendorID ?
										kPGPike_PY_NATDiscovery : kPGPike_PY_Skip,
									kPGPike_PY_VendorID,
									kPGPike_PY_VendorID,
									sendXAuthVendor ? kPGPike_PY_VendorID : kPGPike_PY_Skip,
									kPGPike_PY_Hash,
									kPGPike_PY_None );		CKERR;
						break;
					case kPGPike_AM_XAuth_RespDSS:
					case kPGPike_AM_XAuth_RespRSA:
					case kPGPike_AM_DSS_Sig:
					case kPGPike_AM_RSA_Sig:
						err = pgpIKEDoPacket( &exchange,
									kPGPike_PY_SA,
									kPGPike_PY_KeyExchange,
									kPGPike_PY_Nonce,
									kPGPike_PY_Identification,
									kPGPike_PY_Certificate,
									kPGPike_PY_Signature,
									exchange->rcvdNATTravVendorID ?
										kPGPike_PY_NATDiscovery : kPGPike_PY_Skip,
									kPGPike_PY_VendorID,
									kPGPike_PY_VendorID,
									sendXAuthVendor ? kPGPike_PY_VendorID : kPGPike_PY_Skip,
									kPGPike_PY_None );		CKERR;
						break;
					default:
						pgpAssert( 0 );	/* unsupported */
						break;
				}
				exchange->state = kPGPike_S_AM_WaitFinal;
				break;
			}
			case kPGPike_S_MM_WaitSA:
				exchange->outAlert	= kPGPike_AL_ResponderLifetime;
				exchange->outRLSeconds = exchange->outRLKB = FALSE;
				exchange->outInfoProtocol = kPGPike_PR_IKE;
				exchange->outInfoSPICount = 0;
				if( ( ( exchange->proposals->secLifeTime > exchange->ike->secLifeTimeIKE ) &&
					exchange->ike->secLifeTimeIKE ) ||
					( !exchange->proposals->secLifeTime && exchange->ike->secLifeTimeIKE ) )
				{
					exchange->outRLSeconds	= TRUE;
					sendRL = TRUE;
				}
				if( ( ( exchange->proposals->kbLifeTime > exchange->ike->kbLifeTimeIKE ) &&
					exchange->ike->kbLifeTimeIKE ) ||
					( !exchange->proposals->kbLifeTime && exchange->ike->kbLifeTimeIKE ) )
				{
					exchange->outRLKB		= TRUE;
					sendRL = TRUE;
				}
				err = pgpIKEDoPacket( &exchange, 
							kPGPike_PY_SA,
							sendRL ? kPGPike_PY_Notification : kPGPike_PY_Skip,
							kPGPike_PY_VendorID,
							kPGPike_PY_VendorID,
							( exchange->partner->authStyle != kPGPike_AS_Normal ) ?
								kPGPike_PY_VendorID : kPGPike_PY_Skip,
							exchange->rcvdNATTravVendorID ?
								kPGPike_PY_NATDiscovery : kPGPike_PY_Skip,
							kPGPike_PY_None );CKERR;
				err = pgpIKELoadGroup( exchange );	CKERR;
				exchange->state = kPGPike_S_MM_WaitKE;
				break;
			case kPGPike_S_MM_WaitKE:
				if( ( exchange->initNonceLen == 0 ) || ( IsNull( exchange->dhYi ) ) )
				{
					err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_InvalidPayload );
					goto done;
				}
				switch( exchange->proposals->t[0].u.ike.authMethod )
				{
					case kPGPike_AM_PreSharedKey:
					case kPGPike_AM_DSS_Sig:
					case kPGPike_AM_RSA_Sig:
						sharedKey = FALSE;
						if( exchange->proposals->t[0].u.ike.authMethod == kPGPike_AM_PreSharedKey )
							sharedKey = TRUE;
						err = pgpIKEDoPacket( &exchange,
								kPGPike_PY_KeyExchange,
								kPGPike_PY_Nonce,
								sharedKey ? kPGPike_PY_Skip : kPGPike_PY_CertRequest,
								kPGPike_PY_None );CKERR;
						break;
					default:
						pgpAssert( 0 );	/* unsupported */
						break;
				}
				/* encrypto mondo */
				err = pgpIKEGoSecure( exchange );	CKERR;
				exchange->state = kPGPike_S_MM_WaitFinal;
				break;
			case kPGPike_S_AM_WaitFinal:
			case kPGPike_S_MM_WaitFinal:
				if( partner->authenticated )
				{
					if( exchange->state != kPGPike_S_AM_WaitFinal )
					{
						if( partner->authStyle == kPGPike_AS_HybridAuth )
						{
							switch( exchange->proposals->t[0].u.ike.authMethod )
							{
								case kPGPike_AM_DSS_Sig:
								case kPGPike_AM_RSA_Sig:
									err = pgpIKEDoPacket( &exchange,
											kPGPike_PY_Identification,
											kPGPike_PY_Hash,
											kPGPike_PY_None );	CKERR;
									break;
								default:
									pgpAssert( 0 );	/* unsupported */
									break;
							}
						}
						else switch( exchange->proposals->t[0].u.ike.authMethod )
						{
							case kPGPike_AM_PreSharedKey:
								err = pgpIKEDoPacket( &exchange,
										kPGPike_PY_Identification,
										kPGPike_PY_Hash,
										kPGPike_PY_None );	CKERR;
								break;
							case kPGPike_AM_DSS_Sig:
							case kPGPike_AM_RSA_Sig:
								err = pgpIKEDoPacket( &exchange,
										kPGPike_PY_Identification,
										kPGPike_PY_Certificate,
										kPGPike_PY_Signature,
										kPGPike_PY_None );	CKERR;
								break;
							case kPGPike_AM_RSA_Encrypt:
							case kPGPike_AM_RSA_Encrypt_R:
								err = pgpIKEDoPacket( &exchange,
										kPGPike_PY_Hash,
										kPGPike_PY_None );	CKERR;
								break;
							default:
								pgpAssert( 0 );	/* unsupported */
								break;
						}
					}
					partner->kbLifeTime		= exchange->proposals->kbLifeTime;
					partner->secLifeTime	= exchange->proposals->secLifeTime;
					partner->birthTime		= PGPGetTime();
					if( IsntNull( partner->pgpAuthKey.pass ) )
						PGPFreeData( partner->pgpAuthKey.pass );
					partner->pgpAuthKey.pass = NULL;
					partner->pgpAuthKey.passLength = 0;
					if( IsntNull( partner->x509AuthKey.pass ) )
						PGPFreeData( partner->x509AuthKey.pass );
					partner->x509AuthKey.pass = NULL;
					partner->x509AuthKey.passLength = 0;
					partner->ready			= TRUE;
					err = pgpIKEFreeExchange( exchange );
					exchange = NULL;
					err = pgpIKELocalAlert( partner->ike, partner->ipAddress,
							kPGPike_AL_None, kPGPike_IA_NewPhase1SA, FALSE );
#if PGPIKE_DEBUG
					pgpIKEDebugLog( partner->ike, TRUE, "\tPhase 1 SA Negotiated(R)\n" );
#endif
					if( ( PGPIKE_XAUTHCISCO || ( partner->authStyle == kPGPike_AS_Normal ) )
						&& partner->virtualIP )
					{
						/* if mode-cfg is requested, we initiate a Request transaction */
						err = pgpIKEStartConfigRequest( partner, &exchange );	CKERR;
					}
					if( ( partner->authStyle != kPGPike_AS_Normal ) && !partner->termSchedule )
						partner->termSchedule = PGPGetTime() + kPGPike_XAuthSlack;
				}
				else
				{
					err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_AuthenticationFailed );
					goto done;
				}
				break;
			case kPGPike_S_QM_WaitSA:
				if( exchange->initNonceLen > 0 )
				{
					PGPBoolean				pfs = FALSE,
											idC = FALSE;
					
					exchange->state = kPGPike_S_QM_WaitHash3;
					if( IsntNull( exchange->idBody ) != IsntNull( exchange->idRBody ) )
					{
#if PGPIKE_DEBUG
						pgpIKEDebugLog( exchange->ike, TRUE, "\tOnly one client ID rcvd!\n" );
#endif
						err = pgpIKEAbortExchange( &exchange,
										kPGPike_AL_InvalidID );
						goto done;
					}
					if( IsntNull( exchange->idBody ) )
					{
						PGPUInt32				idIP,
												curLocalIP,
												fullMask;
						PGPBoolean				rrange;
						PGPikeMTClientIDCheck	msg;
						PGPByte					ipProtocol;
						PGPUInt16				ipPort;
						
						/* interpret IDci */
						if( IsPGPError( pgpIKEProcessP2ID(
								exchange->idBody,
								exchange->idBodySize,
								&rrange,
								&idIP,
								&fullMask,
								&ipProtocol,
								&ipPort ) ) )
						{
#if PGPIKE_DEBUG
							pgpIKEDebugLog( exchange->ike, TRUE, "\tClient ID Parse error!\n" );
#endif
							err = pgpIKEAbortExchange( &exchange,
											kPGPike_AL_InvalidID );
							goto done;
						}
#if PGPIKE_DEBUG
						{
							char	ip1Str[20],
									ip2Str[20];
							
							pgpIKEGetIPString( idIP, ip1Str );
							pgpIKEGetIPString( fullMask, ip2Str );
							pgpIKEDebugLog( exchange->ike, TRUE, "\tIDci: %s/%s (prot:%u,port:%u)\n",
								ip1Str, ip2Str, (PGPUInt32)ipProtocol, (PGPUInt32)ipPort );
						}
#endif
						pgpClearMemory( &msg, sizeof(PGPikeMTClientIDCheck) );
						msg.ipAddress		= partner->ipAddress;
						msg.destIsRange		= rrange;
						msg.ipAddrStart		= idIP;
						msg.ipMaskEnd		= fullMask;
						msg.ipProtocol		= ipProtocol;
						msg.ipPort			= ipPort;
						if( !idIP && !fullMask )
							msg.ipAddrStart = 1;
						err = (exchange->ike->msgProc)( (PGPikeContextRef)exchange->ike,
								exchange->ike->userData, kPGPike_MT_ClientIDCheck, &msg );CKERR;
						if( !msg.approved )
						{
#if PGPIKE_DEBUG
							pgpIKEDebugLog( exchange->ike, TRUE, "\tClient ID not approved.\n" );
#endif
							err = pgpIKEAbortExchange( &exchange,
											kPGPike_AL_InvalidID );
							goto done;
						}
						exchange->destination->ipAddrStart		= idIP;
						exchange->destination->ipMaskEnd		= fullMask;
						exchange->destination->destIsRange		= rrange;
						exchange->destination->ipPort			= ipPort;
						exchange->destination->ipProtocol		= ipProtocol;
						/* make sure IDcr is our local IP here */
						/* ***** change this to support local gateway! */
						curLocalIP = partner->assignedIP ? partner->assignedIP : partner->localIPAddress;
						if( IsPGPError( pgpIKEProcessP2ID( exchange->idRBody,
							exchange->idRBodySize, &rrange, &idIP, &fullMask, &ipProtocol, &ipPort ) ) ||
							( idIP != curLocalIP ) || ( fullMask != 0xFFFFFFFF ) || rrange )
						{
#if PGPIKE_DEBUG
							pgpIKEDebugLog( exchange->ike, TRUE, "\tIDcr is not our ID!\n" );
#endif
							err = pgpIKEAbortExchange( &exchange,
											kPGPike_AL_InvalidID );
							goto done;
						}
						(void)PGPFreeData( exchange->idBody );
						(void)PGPFreeData( exchange->idRBody );
						exchange->idBody = NULL;
						exchange->idRBody = NULL;
						exchange->idBodySize = 0;
						exchange->idRBodySize = 0;
						idC = TRUE;
					}
					if( exchange->proposals->t[0].u.ipsec.groupID != kPGPike_GR_None )
					{
						if( IsNull( exchange->dhYi ) )
						{
							err = pgpIKEAbortExchange( &exchange,
											kPGPike_AL_AuthenticationFailed );
							goto done;
						}
						pfs = TRUE;
						err = pgpIKELoadGroup( exchange );	CKERR;
					}
					exchange->outAlert	= kPGPike_AL_ResponderLifetime;
					exchange->outRLSeconds = exchange->outRLKB = FALSE;
					exchange->outInfoProtocol = exchange->proposals->protocol;
					exchange->outInfoSPICount = 1;
					pgpCopyMemory( &exchange->proposals->initSPI, &exchange->outInfoSPI[0],
									sizeof(PGPipsecSPI) );
					if( ( ( exchange->proposals->secLifeTime > exchange->ike->secLifeTimeIPSEC ) &&
						exchange->ike->secLifeTimeIPSEC ) ||
						( !exchange->proposals->secLifeTime && exchange->ike->secLifeTimeIPSEC ) )
					{
						exchange->outRLSeconds	= TRUE;
						sendRL = TRUE;
					}
					if( ( ( exchange->proposals->kbLifeTime > exchange->ike->kbLifeTimeIPSEC ) &&
						exchange->ike->kbLifeTimeIPSEC ) ||
						( !exchange->proposals->kbLifeTime && exchange->ike->kbLifeTimeIPSEC ) )
					{
						exchange->outRLKB		= TRUE;
						sendRL = TRUE;
					}
					err = pgpIKEDoPacket( &exchange,
								kPGPike_PY_SA,
								kPGPike_PY_Nonce,
								pfs ? kPGPike_PY_KeyExchange : kPGPike_PY_Skip,
								idC ? kPGPike_PY_Identification : kPGPike_PY_Skip,
								idC ? kPGPike_PY_Identification : kPGPike_PY_Skip,
								sendRL ? kPGPike_PY_Notification : kPGPike_PY_Skip,
								kPGPike_PY_None );	CKERR;
				}
				else
				{
					err = pgpIKEAbortExchange( &exchange,
									kPGPike_AL_InvalidPayload );
					goto done;
				}
				break;
			case kPGPike_S_QM_WaitHash3:
			{
				if( !exchange->needsCommit )
				{
					/* Phase 2 complete.  Generate the SAs */
					err = pgpIKECompletePhase2( exchange, &sa );				CKERR;
					err = pgpIKESAEstablished( partner, sa );					CKERR;
					err = pgpIKEFreeExchange( exchange );						CKERR;
					exchange = NULL;
				}
				else
				{
					exchange->complete = TRUE;
				}
				break;
			}
			case kPGPike_S_ND_OneState:
				if( ( exchange->exchangeT == kPGPike_EX_Transaction ) &&
					( exchange->numConfigs > 0 ) )
				{
					PGPUInt16			configIndex;
					PGPUInt16			xauthStrLen;
					PGPBoolean			gotRequests = FALSE;
					
					authCheck = PGPNewSecureData( partner->ike->memMgr, sizeof(PGPikeMTAuthCheck),
							kPGPMemoryMgrFlags_Clear );
					if( IsNull( authCheck ) )
					{
						err = kPGPError_OutOfMemory;
						goto done;
					}
					authCheck->gatewayIP = partner->ipAddress;
					authCheck->transactionID = exchange->configs[0].transactionID;
					for( configIndex = 0; IsntNull( exchange ) &&
							( configIndex < exchange->numConfigs ); configIndex++ )
					{
						PGPikeConfigItem *		configItem = &exchange->configs[configIndex];
						
						switch( configItem->configType )
						{
							case kPGPike_CT_Ack:
							case kPGPike_CT_Set:
								if( !configIndex &&
									( configItem->attribute == kPGPike_AT_XAuth_Status ) )
								{
									if( ( configItem->configType == kPGPike_CT_Set ) &&
										( configItem->value == 1 ) )
									{
										PGPipsecSA *simSA;
										
										partner->xAuthenticated = TRUE;
										pgpIKEFreeConfigs( exchange );
										exchange->numConfigs = 1;
										exchange->configs[0].configType		= kPGPike_CT_Ack;
										exchange->configs[0].transactionID	= authCheck->transactionID;
										exchange->configs[0].attribute		= kPGPike_AT_XAuth_Status;
										exchange->configs[0].value			= 1;
										err = pgpIKEDoPacket( &exchange,
													kPGPike_PY_Attribute,
													kPGPike_PY_None );	CKERR;
										err = pgpIKELocalAlert( partner->ike, partner->ipAddress,
												kPGPike_AL_None, kPGPike_IA_XAuthSuccess, FALSE ); CKERR;\
										err = pgpIKEFreeExchange( exchange );		CKERR;
										exchange = NULL;
										/* if this is a rekey and thus we have an existing phase 2,
											this may end up being the completion of the setup,
											and if so we turn off the xauth kill timer for the P1 */
										found = FALSE;
										for( simSA = partner->ike->sa; IsntNull( simSA ); simSA = simSA->nextSA )
										{
											if( simSA->ipAddress == partner->ipAddress )
											{
												found = TRUE;
												break;
											}
										}
										if( found )
											partner->termSchedule = 0;
										if( partner->virtualIP && !partner->assignedIP )
										{
											/* if mode-cfg is requested, we initiate a Request transaction */
											err = pgpIKEStartConfigRequest( partner, NULL );	CKERR;
										}
									}
									else
									{
										if( configItem->configType == kPGPike_CT_Set )
										{
											pgpIKEFreeConfigs( exchange );
											exchange->numConfigs = 1;
											exchange->configs[0].configType		= kPGPike_CT_Ack;
											exchange->configs[0].transactionID	= authCheck->transactionID;
											exchange->configs[0].attribute		= kPGPike_AT_XAuth_Status;
											exchange->configs[0].value			= 0;
											err = pgpIKEDoPacket( &exchange,
														kPGPike_PY_Attribute,
														kPGPike_PY_None );	CKERR;
										}
										err = pgpIKELocalAlert( partner->ike, partner->ipAddress,
												kPGPike_AL_None, kPGPike_IA_XAuthFailed, FALSE ); CKERR;
										err = pgpIKEAbortExchange( &exchange, kPGPike_AL_AuthenticationFailed );
										goto done;
									}
								}
								else
								{
									err = pgpIKEAbortExchange( &exchange, kPGPike_AL_InvalidAttribute );
									goto done;
								}
								break;
							case kPGPike_CT_Request:
								if( configItem->attribute != kPGPike_AT_XAuth_Type )
									gotRequests = TRUE;
								switch( configItem->attribute )
								{
									case kPGPike_AT_XAuth_Type:
										authCheck->xauthType = (PGPikeXAuthType)configItem->value;
										authCheck->includeType = TRUE;
										break;
									case kPGPike_AT_XAuth_Username:
										xauthStrLen = ( configItem->valueSize > kPGPike_XAuthStringLen - 1 ) ?
												kPGPike_XAuthStringLen - 1 : configItem->valueSize;
										if( IsntNull( configItem->valueP ) )
											pgpCopyMemory( configItem->valueP, authCheck->userName, xauthStrLen );
										authCheck->userName[xauthStrLen] = '\0';
										authCheck->useUserName = TRUE;
										break;
									case kPGPike_AT_XAuth_Password:
										xauthStrLen = ( configItem->valueSize > kPGPike_XAuthStringLen - 1 ) ?
												kPGPike_XAuthStringLen - 1 : configItem->valueSize;
										if( IsntNull( configItem->valueP ) )
											pgpCopyMemory( configItem->valueP, authCheck->password, xauthStrLen );
										authCheck->password[xauthStrLen] = '\0';
										authCheck->usePassword = TRUE;
										break;
									case kPGPike_AT_XAuth_Passcode:
										xauthStrLen = ( configItem->valueSize > kPGPike_XAuthStringLen - 1 ) ?
												kPGPike_XAuthStringLen - 1 : configItem->valueSize;
										if( IsntNull( configItem->valueP ) )
											pgpCopyMemory( configItem->valueP, authCheck->passcode, xauthStrLen );
										authCheck->passcode[xauthStrLen] = '\0';
										authCheck->usePasscode = TRUE;
										break;
									case kPGPike_AT_XAuth_Message:
										xauthStrLen = ( configItem->valueSize > kPGPike_XAuthMessageLen - 1 ) ?
												kPGPike_XAuthMessageLen - 1 : configItem->valueSize;
										if( IsntNull( configItem->valueP ) )
											pgpCopyMemory( configItem->valueP, authCheck->message, xauthStrLen );
										authCheck->message[xauthStrLen] = '\0';
										authCheck->useMessage = TRUE;
										break;
									case kPGPike_AT_XAuth_Challenge:
										xauthStrLen = ( configItem->valueSize > kPGPike_XAuthStringLen - 1 ) ?
												kPGPike_XAuthStringLen - 1 : configItem->valueSize;
										if( IsntNull( configItem->valueP ) )
											pgpCopyMemory( configItem->valueP, authCheck->challenge, xauthStrLen );
										authCheck->challengeSize = configItem->valueSize;
										authCheck->useChallenge = TRUE;
										break;
									case kPGPike_AT_XAuth_Domain:
										xauthStrLen = ( configItem->valueSize > kPGPike_XAuthStringLen - 1 ) ?
												kPGPike_XAuthStringLen - 1 : configItem->valueSize;
										if( IsntNull( configItem->valueP ) )
											pgpCopyMemory( configItem->valueP, authCheck->domain, xauthStrLen );
										authCheck->domain[xauthStrLen] = '\0';
										authCheck->useDomain = TRUE;
										break;
									default:
										err = pgpIKEAbortExchange( &exchange, kPGPike_AL_InvalidAttribute );
										goto done;
								}
								break;
							default:
								err = pgpIKEAbortExchange( &exchange, kPGPike_AL_InvalidAttribute );
								goto done;
						}
					}
					if( gotRequests )
					{
						err = (partner->ike->msgProc)( (PGPikeContextRef)partner->ike,
									partner->ike->userData, kPGPike_MT_AuthCheck, authCheck ); CKERR;
					}
				}
				break;
			default:
				break;
		}
	}
done:
	if( IsntNull( authCheck ) )
		(void)PGPFreeData( authCheck );
	if( PGPHMACContextRefIsValid( p2hmac ) )
		(void)PGPFreeHMACContext( p2hmac );
	while( IsntNull( proposal ) )
	{
		PGPikeProposal *	curP = proposal;
		
		proposal = proposal->nextProposal;
		(void)PGPFreeData( curP );
	}
	while( IsntNull( ndPays ) )
	{
		wNDPays = ndPays;
		ndPays = ndPays->nextND;
		(void)PGPFreeData( wNDPays->pay );
		(void)PGPFreeData( wNDPays );
	}
	if( IsntNull( exchange ) &&
		( exchange->exchangeT == kPGPike_EX_Informational ) )
	{
		(void)pgpIKEFreeExchange( exchange );
	}
	return err;	
}

	PGPError
pgpIKEProcessCertRequest(
	PGPikeExchange **				exchangeP,
	PGPByte *						pp,
	PGPSize							payLen )
{
	PGPError						err = kPGPError_NoErr;
	PGPikeExchange *				exchange = *exchangeP;
	PGPikeCertEncoding				ce = (PGPikeCertEncoding)*pp;
	
	if( payLen < 1 )
	{
		err = pgpIKEAbortExchange( &exchange, kPGPike_AL_CertUnavailable );
		goto done;
	}
	switch( ce )
	{
		case kPGPike_CE_PGP:
			exchange->partner->remotePGPVersion = 2;
			break;
		case kPGPike_CE_X509_Sig:
			exchange->partner->remotePGPVersion = 0;
			break;
		default:
			/* we only generate a local alert in this case, because in many cases
				these can be ignored */
			err = pgpIKELocalAlert( exchange->ike, exchange->partner->ipAddress,
						kPGPike_AL_CertUnavailable, kPGPike_IA_None, FALSE );	CKERR;
			break;
	}
	
	/* we ignore the CA DN info */
done:
	return err;
}

	PGPError
pgpIKEProcessSig(
	PGPikeExchange **				exchangeP,
	PGPByte *						pp,
	PGPSize							payLen )
{
	PGPError						err = kPGPError_NoErr;
	PGPikeExchange *				exchange = *exchangeP;
	PGPByte							authHash[kPGPike_MaxHashSize];
	PGPPublicKeyContextRef			pubKeyCon = kInvalidPGPPublicKeyContextRef;
	PGPKeyDBObjRef					cert;
	
	cert = exchange->partner->remoteObj;
	if( !PGPKeyDBObjRefIsValid( cert ) )
	{
		err = pgpIKEAbortExchange( exchangeP, kPGPike_AL_CertUnavailable );
		goto done;
	}
	/* There's only one SIG payload in IKE, it signs HASH_I or HASH_R */
	err = pgpIKEGetAuthHash( exchange, (PGPBoolean)!exchange->initiator, authHash );CKERR;

	err = PGPNewPublicKeyContext( PGPPeekKeyDBObjKey( cert ),
			kPGPPublicKeyMessageFormat_IKE, &pubKeyCon );
	if( IsntPGPError( err ) )
		err = PGPPublicKeyVerifyRaw( pubKeyCon, authHash,
				exchange->partner->hashSize, pp, payLen );
	if( IsPGPError( err ) )
	{
		err = kPGPError_NoErr;		/* dont pass PGPError up to caller */
		err = pgpIKEAbortExchange( exchangeP, kPGPike_AL_AuthenticationFailed );
		goto done;
	}
	/* phew!  if this checks out, P1 is complete */
	exchange->partner->authenticated = TRUE;
done:
	if( PGPPublicKeyContextRefIsValid( pubKeyCon ) )
		(void)PGPFreePublicKeyContext( pubKeyCon );
	return err;
}

	PGPError
pgpIKEProcessCert(
	PGPikeExchange **				exchangeP,
	PGPByte *						pp,
	PGPSize							payLen )
{
	PGPError						err = kPGPError_NoErr;
	PGPUInt32						numKeys;
	PGPByte							certType;
	PGPikeExchange *				exchange = *exchangeP;
	PGPikePartner *					partner = exchange->partner;
	PGPKeyListRef					keyList = kInvalidPGPKeyListRef;
	PGPKeyIterRef					keyIter = kInvalidPGPKeyIterRef;

	certType = *pp++;
	switch( certType )
	{
		case kPGPike_CE_PGP:
		case kPGPike_CE_X509_Sig:
			if( PGPKeyDBRefIsValid( partner->remoteKeyDB ) )
			{
				/* ignore this.  we have received multiple CERT payloads
					which means that a certificate chain being sent.
					we don't support this right now.  If a cert doesn't
					verify from one level, it is bad.  */
#if PGPIKE_DEBUG
				pgpIKEDebugLog( exchange->ike, TRUE,
					"\tReceived extra CERT payload, ignoring\n" );
#endif
				goto done;
			}
			err = PGPImport( exchange->ike->pgpContext,
					&partner->remoteKeyDB,
					PGPOInputBuffer( exchange->ike->pgpContext, pp, payLen - 1 ),
					PGPOX509Encoding( exchange->ike->pgpContext,
						(PGPBoolean) ( certType == kPGPike_CE_X509_Sig ) ),
					PGPOLastOption( exchange->ike->pgpContext ) );
			if( IsntPGPError( err ) )
			{
				PGPKeySetRef	remoteKeySet;
				
				err = PGPNewKeySet( partner->remoteKeyDB, &remoteKeySet ); CKERR;
				err = PGPCountKeys( remoteKeySet, &numKeys );
				/* Get just the one key out of it */
				err = PGPOrderKeySet( remoteKeySet, kPGPKeyOrdering_Any, FALSE, &keyList );	CKERR;
				err = PGPNewKeyIter( keyList, &keyIter ); CKERR;
				err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_Key,
						&partner->remoteObj ); CKERR;
				if( certType == kPGPike_CE_X509_Sig )
				{
					PGPKeyDBObjRef	userID;
					
					err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_UserID,
								&userID );		CKERR;
					err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_Signature,
							&partner->remoteObj );	CKERR;
				}
				PGPFreeKeySet( remoteKeySet );
			}
			if( IsPGPError( err ) || ( numKeys != 1 ) )
			{
				err = pgpIKEAbortExchange( exchangeP, kPGPike_AL_InvalidCert );
				goto done;
			}
			{
				PGPikeMTRemoteCert			rCert;
				
				pgpClearMemory( &rCert, sizeof(PGPikeMTRemoteCert) );
				rCert.ipAddress		= partner->ipAddress;
				rCert.remoteObj		= partner->remoteObj;
				rCert.remoteKeyDB	= partner->remoteKeyDB;
				err = (partner->ike->msgProc)( (PGPikeContextRef)partner->ike,
												partner->ike->userData,
												kPGPike_MT_RemoteCert,
												&rCert );
				if( IsntPGPError( err ) )
				{
					partner->remoteValid = rCert.valid;
					if( !rCert.approved )
					{
						if( PGPKeyListRefIsValid( keyList ) )
							(void)PGPFreeKeyList( keyList );
						keyList = kInvalidPGPKeyListRef;
						if( PGPKeyIterRefIsValid( keyIter ) )
							(void)PGPFreeKeyIter( keyIter );
						keyIter = kInvalidPGPKeyIterRef;
						err = pgpIKEAbortExchange( exchangeP,
										kPGPike_AL_UnsupportedCert );
					}
				}
			}
			break;
		case kPGPike_CE_DNSSig:
		case kPGPike_CE_X509_PKCS7:
		case kPGPike_CE_X509_Exchange:
		case kPGPike_CE_KerberosToken:
		case kPGPike_CE_CRL:
		case kPGPike_CE_ARL:
		case kPGPike_CE_SPKI:
		case kPGPike_CE_X509_Attrib:
		default:
			err = pgpIKEAbortExchange( exchangeP,
							kPGPike_AL_InvalidCertEncoding );
			goto done;
	}
done:
	if( PGPKeyListRefIsValid( keyList ) )
		(void)PGPFreeKeyList( keyList );
	if( PGPKeyIterRefIsValid( keyIter ) )
		(void)PGPFreeKeyIter( keyIter );
	return err;
}

	PGPError
pgpIKEProcessKE(
	PGPikeExchange **				exchangeP,
	PGPByte *						pp,
	PGPSize							payLen )
{
	PGPError						err = kPGPError_NoErr;
	PGPikeExchange *				exchange = *exchangeP;

	if( exchange->initiator )
	{
		if( exchange->dhYr != kPGPInvalidBigNumRef )
		{
			err = pgpIKEAbortExchange( exchangeP,
							kPGPike_AL_InvalidPayload );
			goto done;
		}
		err = PGPNewBigNum( exchange->ike->pgpContext, TRUE,	
							&exchange->dhYr ); CKERR;
		err = PGPBigNumInsertBigEndianBytes( exchange->dhYr, pp,
											0, payLen ); CKERR;
	}
	else
	{
		if( exchange->dhYi != kPGPInvalidBigNumRef )
		{
			err = pgpIKEAbortExchange( exchangeP,
							kPGPike_AL_InvalidPayload );
			goto done;
		}
		err = PGPNewBigNum( exchange->ike->pgpContext, TRUE,
							&exchange->dhYi ); CKERR;
		err = PGPBigNumInsertBigEndianBytes( exchange->dhYi, pp,
											0, payLen ); CKERR;
	}
done:
	return err;
}

	PGPError
pgpIKEProcessConfig(
	PGPikeExchange **				exchangeP,
	PGPByte *						pp,
	PGPSize							payLen )
{
	PGPError						err = kPGPError_NoErr;
	PGPikeExchange *				exchange = *exchangeP;
	PGPByte *						ep = pp + payLen;
	PGPikeConfigType				configType = (PGPikeConfigType)*pp++;
	PGPUInt16						transactionID;
	PGPikeAttributeType				attribute;
	PGPUInt32						value;
	PGPByte *						valueP;
	PGPUInt16						valueSize;
	PGPBoolean						added;
	
	pgpIKEFreeConfigs( exchange );
	switch( configType )
	{
		case kPGPike_CT_Request:
		case kPGPike_CT_Reply:
		case kPGPike_CT_Set:
		case kPGPike_CT_Ack:
			break;
		default:
#if PGPIKE_DEBUG
			pgpIKEDebugLog( exchange->ike, TRUE, "\tReceived unsupported config type\n" );
#endif
			goto done;
	}
	pp++;
	transactionID = PGPEndianToUInt16( kPGPBigEndian, pp );
	pp += sizeof(PGPUInt16);
	while( ( ep - pp ) >= 4 )
	{
		added = FALSE;
		if( exchange->numConfigs == kPGPike_MaxConfigItems )
		{
#if PGPIKE_DEBUG
			pgpIKEDebugLog( exchange->ike, TRUE, "\tExceeded maximum configs\n" );
#endif
			break;
		}
		pgpIKEGetAttribute( exchange, &pp, ep, &attribute, &value, &valueP, &valueSize );
		switch( attribute )
		{
			case kPGPike_AT_Config_IPv4Address:
			case kPGPike_AT_Config_IPv4Mask:
			case kPGPike_AT_Config_IPv4DNS:
			case kPGPike_AT_Config_IPv4NBNS:
			case kPGPike_AT_Config_IPv4DHCP:
				/* correct byte order of value */
				pgpCopyMemory( pp - sizeof(PGPUInt32), &value, sizeof(PGPUInt32) );
			case kPGPike_AT_Config_AddressExpiry:
			case kPGPike_AT_XAuth_Type:
			case kPGPike_AT_XAuth_Status:
				if( IsNull( valueP ) )
					added = TRUE;
				break;
			case kPGPike_AT_XAuth_Username:
			case kPGPike_AT_XAuth_Password:
			case kPGPike_AT_XAuth_Passcode:
			case kPGPike_AT_XAuth_Message:
			case kPGPike_AT_XAuth_Challenge:
			case kPGPike_AT_XAuth_Domain:
				/* put a hard limit on valueSize here to make sure it doesn't cause
					problems later */
				if( valueSize > kPGPike_XAuthMessageLen - 1 )
					valueSize = kPGPike_XAuthMessageLen - 1;
				added = TRUE;
				break;
			default:
				/*	For config attributes we don't support,
					we ignore them and do not ACK them. */
#if PGPIKE_DEBUG
				pgpIKEDebugLog( exchange->ike, TRUE, "\tConfig: Unsupported Attribute Type (%s)\n", 
					pgpIKEConfigAttrString( attribute ) );
#endif
				if( IsntNull( valueP ) )
					(void)PGPFreeData( valueP );
				break;
		}
		if( added )
		{
			exchange->configs[exchange->numConfigs].configType		= configType;
			exchange->configs[exchange->numConfigs].transactionID	= transactionID;
			exchange->configs[exchange->numConfigs].attribute		= attribute;
			exchange->configs[exchange->numConfigs].value			= value;
			exchange->configs[exchange->numConfigs].valueP			= valueP;
			exchange->configs[exchange->numConfigs].valueSize		= valueSize;
			
			exchange->numConfigs++;
#if PGPIKE_DEBUG
			{
				char *		ctStr = "unknown";
				char		xauthStr[kPGPike_XAuthMessageLen];
				
				pgpIKEDebugLog( exchange->ike, TRUE, "\t\tConfig: (%d) ", transactionID );
				switch( configType )
				{
					case kPGPike_CT_Request:
						ctStr = "Request";
						break;
					case kPGPike_CT_Reply:
						ctStr = "Reply";
						break;
					case kPGPike_CT_Set:
						ctStr = "Set";
						break;
					case kPGPike_CT_Ack:
						ctStr = "Ack";
						break;
					default:
						break;
				}
				switch( attribute )
				{
					case kPGPike_AT_Config_IPv4Address:
					case kPGPike_AT_Config_IPv4Mask:
					case kPGPike_AT_Config_IPv4DNS:
					case kPGPike_AT_Config_IPv4NBNS:
					case kPGPike_AT_Config_IPv4DHCP:
						{
							char	ip1Str[20];
							
							pgpIKEGetIPString( value, ip1Str );
							pgpIKEDebugLog( exchange->ike, FALSE, "%s %s: %s\n", 
								ctStr, pgpIKEConfigAttrString( attribute ), ip1Str );
						}
						break;
					case kPGPike_AT_Config_AddressExpiry:
					case kPGPike_AT_XAuth_Type:
					case kPGPike_AT_XAuth_Status:
						pgpIKEDebugLog( exchange->ike, FALSE, "%s %s: %u\n", 
							ctStr, pgpIKEConfigAttrString( attribute ), value );
						break;
					case kPGPike_AT_XAuth_Challenge:
						pgpIKEDebugData( exchange->ike, "Request xauthChallenge:", valueP, valueSize );
						break;
					case kPGPike_AT_XAuth_Username:
					case kPGPike_AT_XAuth_Password:
					case kPGPike_AT_XAuth_Passcode:
					case kPGPike_AT_XAuth_Message:
					case kPGPike_AT_XAuth_Domain:
						if( IsntNull( valueP ) )
							pgpCopyMemory( valueP, xauthStr, valueSize );
						xauthStr[valueSize] = '\0';
						pgpIKEDebugLog( exchange->ike, FALSE, "%s %s: %s\n", 
							ctStr, pgpIKEConfigAttrString( attribute ), xauthStr );
						break;
					default:
						break;
				}
			}
#endif
		}
	}
done:
	return err;
}

	PGPError
pgpIKEProcessP2ID(
	PGPByte *				inIDBody,
	PGPSize					inIDBodySize,
	PGPBoolean *			outIsRange,
	PGPUInt32 *				ipStart,
	PGPUInt32 *				ipEnd,
	PGPByte *				ipProtocol,
	PGPUInt16 *				ipPort )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				pp = inIDBody;
	PGPipsecIdentity		idType;
	
	*outIsRange = FALSE;
	*ipStart = 0;
	*ipEnd = 0;
	idType = (PGPipsecIdentity) *pp++;
	*ipProtocol = *pp++;
	*ipPort = PGPEndianToUInt16( kPGPBigEndian, pp );
	pp += sizeof(PGPUInt16);
	switch( idType )
	{
		case kPGPike_ID_IPV4_Addr:
			*ipStart = *(PGPUInt32 *)pp;
			pp += sizeof(PGPUInt32);
			*ipEnd = 0xFFFFFFFF;
			break;
		case kPGPike_ID_IPV4_Addr_Range:
			*outIsRange = TRUE;
		case kPGPike_ID_IPV4_Addr_Subnet:
			*ipStart = *(PGPUInt32 *)pp;
			pp += sizeof(PGPUInt32);
			*ipEnd = *(PGPUInt32 *)pp;
			pp += sizeof(PGPUInt32);
			break;
		default:
			err = kPGPError_ItemNotFound;
			break;
	}
	if( IsntPGPError( err ) && ( inIDBody + inIDBodySize ) != pp )
		err = kPGPError_BufferTooSmall;
	return err;
}

	PGPError
pgpIKESelectProposal(
	PGPikeExchange *				exchange,
	PGPikeProposal *				proposal,
	PGPBoolean *					isValid )
{
	PGPError						err = kPGPError_NoErr;
	PGPBoolean						match = FALSE;
	PGPikeProposal *				curR;
	PGPUInt16						trInx;

	curR = proposal;
	if( exchange->initiator )
	{
		PGPikeProposal *			curLR;
		PGPikeProposal *			lastLR;
		
		/*	We should have received back a matching set of
			proposals.  This could be more than one proposal
			if we sent a pair or triplet proposal. */
		
		for( curLR = exchange->proposals; IsntNull( curLR );
				curLR = curLR->nextProposal )
		{
			/* Two lines here had to be removed due to lame IKE
				implementations that don't correctly number their
				return proposals. */
		multiProposal:
			if( ( curR->protocol == curLR->protocol ) &&
				( curR->numTransforms == 1 ) &&
				( curR->kbLifeTime == curLR->kbLifeTime ) &&
				( curR->secLifeTime == curLR->secLifeTime ) )
			{
				for( trInx = 0; trInx < curLR->numTransforms; trInx++ )
				{
					if( pgpIKEIsTransformEqual( curR->protocol,
							&curR->t[0], &curLR->t[trInx] ) )
					{
						match = TRUE;
						break;
					}
					else
						curR = proposal;
				}
				if( match )
				{
					*(PGPUInt32 *)curR->initSPI =
							*(PGPUInt32 *)curLR->initSPI;
					if( IsntNull( curR->nextProposal ) )
					{
						/* Note there is a bug here.  If the remote re-orders a
							multiple proposal from us, we will not accept it. */
						match = FALSE;
						if( IsntNull( curLR->nextProposal ) )
						{
							curR = curR->nextProposal;
							curLR = curLR->nextProposal;
							goto multiProposal;
						}
						else
							goto noMatch;
					}
					if( IsntNull( curLR->nextProposal ) &&
						( curLR->nextProposal->number == curLR->number ) )
						goto noMatch;
					/* We have a positive match */
					curLR = exchange->proposals;
					while( IsntNull( curLR ) )
					{
						lastLR = curLR;
						curLR = curLR->nextProposal;
						(void)PGPFreeData( lastLR );
					}
					exchange->proposals = NULL;
					if( ( exchange->state == kPGPike_S_MM_WaitSA ) ||
						( exchange->state == kPGPike_S_AM_WaitSA ) )
					{
#if PGPIKE_DEBUG
						pgpIKEDebugLog( exchange->ike, TRUE, "\tProposal Selected (I): %s, %s\n",
							pgpIKEAuthMethodString( proposal->t[0].u.ike.authMethod ),
							pgpIKECipherString( proposal->t[0].u.ike.cipher ) );
#endif
						switch( proposal->t[0].u.ike.authMethod )
						{
							case kPGPike_AM_XAuth_InitPreShared:
								proposal->t[0].u.ike.authMethod = kPGPike_AM_PreSharedKey;
								break;
							case kPGPike_AM_XAuth_InitDSS:
							case kPGPike_AM_HAuth_InitDSS:
								proposal->t[0].u.ike.authMethod = kPGPike_AM_DSS_Sig;
								break;
							case kPGPike_AM_XAuth_InitRSA:
							case kPGPike_AM_HAuth_InitRSA:
								proposal->t[0].u.ike.authMethod = kPGPike_AM_RSA_Sig;
								break;
							default:
								break;
						}
					}
					curR = NULL;
					for( curLR = proposal; IsntNull( curLR ); curLR = curLR->nextProposal )
					{
						if( IsNull( exchange->proposals ) )
						{
							exchange->proposals = PGPNewData( exchange->ike->memMgr,
								sizeof(PGPikeProposal), kPGPMemoryMgrFlags_Clear );
							if( IsNull( exchange->proposals ) )
							{
								err = kPGPError_OutOfMemory;
								goto done;
							}
							curR = exchange->proposals;
						}
						else
						{
							curR->nextProposal = PGPNewData( exchange->ike->memMgr,
								sizeof(PGPikeProposal), kPGPMemoryMgrFlags_Clear );
							if( IsNull( curR->nextProposal ) )
							{
								err = kPGPError_OutOfMemory;
								goto done;
							}
							curR = curR->nextProposal;
						}
						pgpCopyMemory( curLR, curR, sizeof( PGPikeProposal ) );
					}
					if( IsntNull( curR ) )
						curR->nextProposal = NULL;
					break;
				}
			}
			else
			{
				PGPUInt8	curPNum = curLR->number;
				
		noMatch:
				curR = proposal;
				while( IsntNull( curLR->nextProposal ) &&
						( curLR->nextProposal->number == curPNum ) )
					curLR = curLR->nextProposal;
			}
		}
	}
	else
	{
		PGPikeProposal *			acceptProposal = NULL;
		PGPUInt16					acceptCount = 0;
		PGPInt16					lastPNum = -1;
		
		/*	we must now select a proposal from the set */
		
#if PGPIKE_DEBUG
		PGPikeProposal *				dbgR = proposal;
		
		pgpIKEDebugLog( exchange->ike, TRUE, "\tRemote Proposals:\n" );
		if( ( exchange->state == kPGPike_S_MM_WaitSA ) ||
			( exchange->state == kPGPike_S_AM_WaitSA ) )
		{
			for( trInx = 0; trInx < dbgR->numTransforms; trInx++ )
			{
				pgpIKEDebugLog( exchange->ike, FALSE, "\t\tIKE: (%d) %s, %s, %s, %s, S:%u, KB:%u\n",
					trInx, pgpIKEAuthMethodString( dbgR->t[trInx].u.ike.authMethod ),
					pgpIKECipherString( dbgR->t[trInx].u.ike.cipher ),
					pgpIKEHashTypeString( dbgR->t[trInx].u.ike.hash ),
					pgpIKEGroupTypeString( dbgR->t[trInx].u.ike.groupID ),
					dbgR->secLifeTime, dbgR->kbLifeTime );
			}
		}
		else
		{
			while( IsntNull( dbgR ) )
			{
				pgpIKEDebugLog( exchange->ike, FALSE, "\t\tIPSEC: (%d) ", dbgR->number );
				switch( dbgR->protocol )
				{
					case kPGPike_PR_AH:
						pgpIKEDebugLog( exchange->ike, FALSE, "AH: %s, %s",
							pgpIKEipsecAuthTypeString( dbgR->t[0].u.ipsec.ah.authAttr ),
							( dbgR->t[0].u.ipsec.ah.mode == kPGPike_PM_Tunnel ) ?
							"Tunnel" : "Transport" );
						break;
					case kPGPike_PR_ESP:
						pgpIKEDebugLog( exchange->ike, FALSE, "ESP: %s, %s, %s, %s%u",
							pgpIKEipsecAuthTypeString( dbgR->t[0].u.ipsec.esp.authAttr ),
							pgpIKEipsecCipherTypeString( dbgR->t[0].u.ipsec.esp.cipher ),
							( dbgR->t[0].u.ipsec.esp.mode == kPGPike_PM_Tunnel ) ?
							"Tunnel" : "Transport", dbgR->t[0].u.ipsec.groupID ?
							"PFS-" : "No PFS-", dbgR->t[0].u.ipsec.groupID ?
							(PGPUInt32)dbgR->t[0].u.ipsec.groupID : 0 );
						break;
					case kPGPike_PR_IPCOMP:
						pgpIKEDebugLog( exchange->ike, FALSE, "IPCOMP" );
						break;
					default:
						break;
				}
				pgpIKEDebugLog( exchange->ike, FALSE, ", S:%u, KB:%u\n",
					dbgR->secLifeTime, dbgR->kbLifeTime );
				dbgR = dbgR->nextProposal;
			}
		}
#endif
		pgpAssert( IsNull( exchange->proposals ) );
		
		while( IsntNull( curR ) )
		{
			for( trInx = 0; trInx < curR->numTransforms; trInx++ )
			{
				if( pgpIKEIsTransformValid( exchange->partner, curR->protocol,
							&curR->t[trInx] ) )
				{
					/* good transform, select it */
					pgpCopyMemory( &curR->t[trInx], &curR->t[0],
									sizeof(PGPikeGenericTransform) );
					if( ( exchange->state == kPGPike_S_MM_WaitSA ) ||
						( exchange->state == kPGPike_S_AM_WaitSA ) )
					{
#if PGPIKE_DEBUG
						pgpIKEDebugLog( exchange->ike, TRUE, "\tProposal Selected (R): %s, %s\n",
							pgpIKEAuthMethodString( curR->t[0].u.ike.authMethod ),
							pgpIKECipherString( curR->t[0].u.ike.cipher ) );
#endif
					}
					break;
				}
			}
			if( trInx >= curR->numTransforms )
			{
				if( IsntNull( acceptProposal ) && ( acceptProposal->number == curR->number ) )
				{
					/* partial rejection of SA bundle proposal */
					while( IsntNull( acceptProposal ) )
					{
						PGPikeProposal *	fProp = acceptProposal;
						
						acceptProposal = acceptProposal->nextProposal;
						(void)PGPFreeData( fProp );
						acceptCount--;
					}
				}
				goto unsupportedProposal;
			}
			
			curR->numTransforms = 1;
			if( IsntNull( acceptProposal ) )
			{
				if( acceptProposal->number == curR->number )
				{
					PGPikeProposal *	walkP;
					
					if( acceptCount >= kPGPike_MaxTransforms )
						goto unsupportedProposal;
					for( walkP = acceptProposal; IsntNull( walkP->nextProposal );
						walkP = walkP->nextProposal )
						;
					walkP->nextProposal = PGPNewData( exchange->ike->memMgr,
									sizeof(PGPikeProposal), kPGPMemoryMgrFlags_Clear );
					if( IsNull( walkP->nextProposal ) )
					{
						err = kPGPError_OutOfMemory;
						goto done;
					}
					walkP = walkP->nextProposal;
					pgpCopyMemory( curR, walkP, sizeof( PGPikeProposal ) );
					walkP->nextProposal = NULL;
					acceptCount++;
				}
				else
					break;
			}
			else
			{
				if( lastPNum == (PGPInt16)curR->number )
					goto unsupportedProposal;
				acceptProposal = PGPNewData( exchange->ike->memMgr,
					sizeof(PGPikeProposal), kPGPMemoryMgrFlags_Clear );
				if( IsNull( acceptProposal ) )
				{
					err = kPGPError_OutOfMemory;
					goto done;
				}
				pgpCopyMemory( curR, acceptProposal, sizeof( PGPikeProposal ) );
				acceptProposal->nextProposal = NULL;
				acceptCount++;
			}
			
		unsupportedProposal:
			lastPNum = curR->number;
			curR = curR->nextProposal;
		}
		if( IsntNull( acceptProposal ) )
		{
			PGPikeProposal *	curP;
			
			if( exchange->state == kPGPike_S_QM_WaitSA )
			{
				PGPBoolean	hasESP		= FALSE,
							hasCipher	= FALSE,
							hasAH		= FALSE,
							hasIPCOMP	= FALSE,
							badCompound	= FALSE;
				
				for( curP = acceptProposal; IsntNull( curP ) && !badCompound;
						curP = curP->nextProposal )
				{
					if( curP->protocol == kPGPike_PR_IPCOMP )
					{
						if( !hasIPCOMP )
							hasIPCOMP = TRUE;
						else
							badCompound = TRUE;
					}
					else if( curP->protocol == kPGPike_PR_ESP )
					{
						if( !hasESP )
						{
							hasESP = TRUE;
							if( curP->t[0].u.ipsec.esp.cipher != kPGPike_ET_NULL )
								hasCipher = TRUE;
						}
						else
							badCompound = TRUE;
					}
					else if( curP->protocol == kPGPike_PR_AH )
					{
						if( !hasAH )
							hasAH = TRUE;
						else
							badCompound = TRUE;
					}
				}
				if( !hasCipher && !exchange->ike->allowedAlgorithms.espNULL )
					badCompound = TRUE;
				if( badCompound )
				{
					while( IsntNull( acceptProposal ) )
					{
						curP = acceptProposal;
						
						acceptProposal = acceptProposal->nextProposal;
						(void)PGPFreeData( curP );
						acceptCount--;
					}
					if( IsntNull( curR ) )
						goto unsupportedProposal;
				}
				
				for( curP = acceptProposal; IsntNull( curP ); curP = curP->nextProposal )
				{
					err = PGPContextGetRandomBytes(
							exchange->ike->pgpContext,
							&curP->respSPI,
							sizeof(PGPipsecSPI) ); CKERR;
				}
			}
			if( IsntNull( acceptProposal ) )
			{
				match = TRUE;
				exchange->proposals = acceptProposal;
			}
		}
	}
done:
	*isValid = match;
	return err;
}

	PGPError
pgpIKEProcessProposal(
	PGPikeExchange **		exchangeP,
	PGPByte *				mp,
	PGPByte *				ep,
	PGPikeAlert	*			alert,
	PGPikeProposal **		proposal )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeProposal *		prop = NULL;
	PGPikeExchange *		exchange = *exchangeP;

	*alert = kPGPike_AL_None;
	*proposal = NULL;
	prop = PGPNewData( exchange->ike->memMgr, sizeof(PGPikeProposal),
							kPGPMemoryMgrFlags_Clear );
	if( IsntNull( prop ) )
	{
		PGPUInt8				spiSize;
		PGPUInt8				transIndex;
		PGPBoolean				zeroBased = FALSE;
		PGPUInt8				numTransforms;
		
		if( ep - mp < sizeof(PGPUInt32) )
		{
			*alert = kPGPike_AL_UnequalPayloadLengths;
			goto done;
		}
		prop->number			= *mp++;
		prop->protocol			= (PGPipsecProtocol)*mp++;
		spiSize					= *mp++;
		numTransforms			= *mp++;
		
		switch( prop->protocol )
		{
			case kPGPike_PR_IKE:
				if( spiSize > 16 )
				{
					*alert = kPGPike_AL_InvalidSPI;
					goto done;
				}
				break;
			case kPGPike_PR_IPCOMP:
				if( spiSize == sizeof( PGPipsecSPI ) )
				{
					if( exchange->initiator )
						pgpCopyMemory( mp, &prop->respSPI, spiSize );
					else
						pgpCopyMemory( mp, &prop->initSPI, spiSize );
				}
				else if( spiSize == sizeof( PGPUInt16 ) )
				{
					PGPUInt16	pcpCPI16 = PGPEndianToUInt16( kPGPBigEndian, mp );
					
					if( exchange->initiator )
					{
						prop->respSPI[2] = pcpCPI16 >> 8;
						prop->respSPI[3] = pcpCPI16 & 0xFF;
					}
					else
					{
						prop->initSPI[2] = pcpCPI16 >> 8;
						prop->initSPI[3] = pcpCPI16 & 0xFF;
					}
				}
				else
				{
					*alert = kPGPike_AL_InvalidSPI;
					goto done;
				}
				break;
			case kPGPike_PR_AH:
			case kPGPike_PR_ESP:
				if( spiSize != sizeof( PGPipsecSPI ) )
				{
					*alert = kPGPike_AL_InvalidSPI;
					goto done;
				}
				if( exchange->initiator )
					pgpCopyMemory( mp, &prop->respSPI, spiSize );
				else
					pgpCopyMemory( mp, &prop->initSPI, spiSize );
				break;
			default:
				*alert = kPGPike_AL_InvalidProtocolID;
				goto done;
		}
		if( ep - mp < spiSize )
		{
			*alert = kPGPike_AL_UnequalPayloadLengths;
			goto done;
		}
		mp += spiSize;
		
		/* Transforms */
		for( transIndex = 0; transIndex < numTransforms; transIndex++ )
		{
			PGPByte *				tep;
			PGPUInt16				transLen;
			PGPikeAttributeType		attribute;
			PGPUInt32				value;
			PGPikeGenericTransform *trx = &prop->t[prop->numTransforms];
			
			if( ep - mp < kPGPike_ISAKMPPayloadHeaderSize )
			{
				*alert = kPGPike_AL_UnequalPayloadLengths;
				goto done;
			}
			mp++;	mp++;
			transLen = PGPEndianToUInt16( kPGPBigEndian, mp );
			mp+= sizeof(PGPUInt16);
			tep = mp + transLen - kPGPike_ISAKMPPayloadHeaderSize;
			if( ep < tep )
			{
				*alert = kPGPike_AL_UnequalPayloadLengths;
				goto done;
			}
			if( transIndex < kPGPike_MaxPropTransforms )
			{
				if( !transIndex && ( *mp == 0 ) )
					zeroBased = TRUE;
				if( ( *mp++ == transIndex + ( zeroBased ? 0 : 1 ) ) ||
					( exchange->initiator && ( numTransforms == 1 ) ) )
				{
					PGPUInt16	transformID;
					
					transformID = *mp++;
					mp++; mp++;	/* reserved */
					switch( prop->protocol )
					{
						case kPGPike_PR_IKE:
							if( transformID != kPGPike_Key_IKE_Transform )
							{
								*alert = kPGPike_AL_InvalidTransform;
								goto done;
							}
							trx->u.ike.modpGroup		= TRUE;	/* default */
							trx->u.ike.groupID			= kPGPike_GR_MODPOne;
							while( ( tep - mp ) >= 4 )
							{
								pgpIKEGetAttribute( exchange, &mp, tep, &attribute, &value, NULL, NULL );
								switch( attribute )
								{
									case kPGPike_AT_AuthMethod:
										trx->u.ike.authMethod =
											(PGPikeAuthMethod)value;
										break;
									case kPGPike_AT_GroupType:
										if( value != kPGPike_MODPGroupType )
											goto unsupportedTransform;
										trx->u.ike.modpGroup = TRUE;
										break;
									case kPGPike_AT_GroupDesc:
										/* if unspecified, MODP one is used */
										switch( value )
										{
											case kPGPike_GR_MODPOne:
											case kPGPike_GR_MODPTwo:
											case kPGPike_GR_MODPFive:
												trx->u.ike.groupID =
													(PGPikeGroupID)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_EncryptAlg:
										switch( value )
										{
											case kPGPike_SC_3DES_CBC:
											case kPGPike_SC_CAST_CBC:
											case kPGPike_SC_DES_CBC:
												trx->u.ike.cipher =
													(PGPikeCipher)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_HashAlg:
										switch( value )
										{
											case kPGPike_HA_SHA1:
											case kPGPike_HA_MD5:
												trx->u.ike.hash = (PGPikeHash)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_LifeType:
									{
										PGPikeLifeType	lifeType =
															(PGPikeLifeType) value;
										if( ( (tep - mp ) >= 4 ) &&
											( ( lifeType == kPGPike_LT_Seconds ) ||
											( lifeType == kPGPike_LT_Kilobytes ) ) )
										{
											pgpIKEGetAttribute( exchange, &mp, tep, &attribute,
												&value, NULL, NULL );
											if( attribute == kPGPike_AT_LifeDuration )
											{
												if( lifeType == kPGPike_LT_Seconds )
													prop->secLifeTime	= value;
												else
													prop->kbLifeTime	= value;
											}
											else
											{
												*alert = kPGPike_AL_InvalidTransform;
												goto done;
											}
										}
										else
										{
											*alert = kPGPike_AL_InvalidTransform;
											goto done;
										}
										break;
									}
									default:
#if PGPIKE_DEBUG
										pgpIKEDebugLog( exchange->ike, TRUE,
											"\tReceived unsupported P1 attribute: %s\n",
											pgpIKEP1AttributeString( attribute ) );
#endif
										*alert = kPGPike_AL_InvalidAttribute;
										goto done;
								}
							}
							if( ( trx->u.ike.authMethod == kPGPike_AM_None ) ||
								( trx->u.ike.hash		== kPGPike_HA_None ) ||
								( trx->u.ike.cipher		== kPGPike_SC_None ) ||
								!trx->u.ike.modpGroup )
							{
								*alert = kPGPike_AL_InvalidTransform;
								goto done;
							}
							prop->numTransforms++;
							break;
						case kPGPike_PR_AH:
						{
							PGPipsecAHTransformID	ahTID;
							
							switch( ahTID = (PGPipsecAHTransformID)transformID )
							{
								case kPGPike_AH_MD5:
								case kPGPike_AH_SHA:
									trx->u.ipsec.ah.authAlg = ahTID;
									break;
								default:
									goto unsupportedTransform;
							}
							trx->u.ipsec.ah.mode = kPGPike_PM_Transport;
							while( ( tep - mp ) >= 4 )
							{
								pgpIKEGetAttribute( exchange, &mp, tep, &attribute, &value, NULL, NULL );
								switch( attribute )
								{
									case kPGPike_AT_IPSEC_AuthAttr:
										switch( value )
										{
											case kPGPike_AA_HMAC_MD5:
											case kPGPike_AA_HMAC_SHA:
												trx->u.ipsec.ah.authAttr =
													(PGPipsecAuthAttribute)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_IPSEC_Encapsulation:
										switch( value )
										{
											case kPGPike_PM_Tunnel:
											case kPGPike_PM_Transport:
												trx->u.ipsec.ah.mode =
													(PGPipsecEncapsulation)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_IPSEC_GroupDesc:
										switch( value )
										{
											case kPGPike_GR_MODPOne:
											case kPGPike_GR_MODPTwo:
											case kPGPike_GR_MODPFive:
												trx->u.ipsec.groupID =
													(PGPikeGroupID)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_IPSEC_LifeType:
									{
										PGPikeLifeType	lifeType =
															(PGPikeLifeType) value;
										if( ( (tep - mp ) >= 4 ) &&
											( ( lifeType == kPGPike_LT_Seconds ) ||
											( lifeType == kPGPike_LT_Kilobytes ) ) )
										{
											pgpIKEGetAttribute( exchange, &mp, tep, &attribute,
																&value, NULL, NULL );
											if( attribute == kPGPike_AT_IPSEC_Duration )
											{
												if( lifeType == kPGPike_LT_Seconds )
													prop->secLifeTime	= value;
												else
													prop->kbLifeTime	= value;
											}
											else
											{
												*alert = kPGPike_AL_InvalidTransform;
												goto done;
											}
										}
										else
										{
											*alert = kPGPike_AL_InvalidTransform;
											goto done;
										}
										break;
									}
									default:
#if PGPIKE_DEBUG
										pgpIKEDebugLog( exchange->ike, TRUE,
											"\tReceived unsupported AH P2 attribute: %s\n",
											pgpIKEP2AttributeString( attribute ) );
#endif
										*alert = kPGPike_AL_InvalidAttribute;
										goto done;
								}
							}
							if( ( trx->u.ipsec.ah.mode == kPGPike_PM_None ) ||
								( trx->u.ipsec.ah.authAttr == kPGPike_AA_None ) )
							{
								*alert = kPGPike_AL_InvalidTransform;
								goto done;
							}
							prop->numTransforms++;
							break;
						}
						case kPGPike_PR_ESP:
						{
							PGPipsecESPTransformID	espTID;
							
							switch( espTID = (PGPipsecESPTransformID)transformID )
							{
								case kPGPike_ET_NULL:
								case kPGPike_ET_3DES:
								case kPGPike_ET_CAST:
								case kPGPike_ET_DES_IV64:
								case kPGPike_ET_DES:
									trx->u.ipsec.esp.cipher = espTID;
									break;
								default:
									goto unsupportedTransform;
							}
							trx->u.ipsec.esp.mode = kPGPike_PM_Transport;
							while( ( tep - mp ) >= 4 )
							{
								pgpIKEGetAttribute( exchange, &mp, tep, &attribute, &value, NULL, NULL );
								switch( attribute )
								{
									case kPGPike_AT_IPSEC_AuthAttr:
										switch( value )
										{
											case kPGPike_AA_HMAC_MD5:
											case kPGPike_AA_HMAC_SHA:
												trx->u.ipsec.esp.authAttr =
													(PGPipsecAuthAttribute)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_IPSEC_Encapsulation:
										switch( value )
										{
											case kPGPike_PM_Tunnel:
											case kPGPike_PM_Transport:
												trx->u.ipsec.esp.mode =
													(PGPipsecEncapsulation)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_IPSEC_GroupDesc:
										switch( value )
										{
											case kPGPike_GR_MODPOne:
											case kPGPike_GR_MODPTwo:
											case kPGPike_GR_MODPFive:
												trx->u.ipsec.groupID =
													(PGPikeGroupID)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_IPSEC_LifeType:
									{
										PGPikeLifeType	lifeType =
															(PGPikeLifeType) value;
										if( ( ( tep - mp ) >= 4 ) &&
											( ( lifeType == kPGPike_LT_Seconds ) ||
											( lifeType == kPGPike_LT_Kilobytes ) ) )
										{
											pgpIKEGetAttribute( exchange, &mp, tep, &attribute,
																&value, NULL, NULL );
											if( attribute == kPGPike_AT_IPSEC_Duration )
											{
												if( lifeType == kPGPike_LT_Seconds )
													prop->secLifeTime	= value;
												else
													prop->kbLifeTime	= value;
											}
											else
											{
												*alert = kPGPike_AL_InvalidTransform;
												goto done;
											}
										}
										else
										{
											*alert = kPGPike_AL_InvalidTransform;
											goto done;
										}
										break;
									}
									default:
#if PGPIKE_DEBUG
										pgpIKEDebugLog( exchange->ike, TRUE,
											"\tReceived unsupported ESP P2 attribute: %s\n",
											pgpIKEP2AttributeString( attribute ) );
#endif
										*alert = kPGPike_AL_InvalidAttribute;
										goto done;
								}
							}
							if( ( trx->u.ipsec.esp.mode == kPGPike_PM_None ) )
							{
								*alert = kPGPike_AL_InvalidTransform;
								goto done;
							}
							prop->numTransforms++;
							break;
						}
						case kPGPike_PR_IPCOMP:
						{
							PGPipsecIPCOMPTransformID	ipcompTID;
							
							switch( ipcompTID = (PGPipsecIPCOMPTransformID)transformID )
							{
								case kPGPike_IC_Deflate:
								case kPGPike_IC_LZS:
									trx->u.ipsec.ipcomp.compAlg = ipcompTID;
									break;
								default:
									goto unsupportedTransform;
							}
							while( ( tep - mp ) >= 4 )
							{
								pgpIKEGetAttribute( exchange, &mp, tep, &attribute, &value, NULL, NULL );
								switch( attribute )
								{
									case kPGPike_AT_IPSEC_GroupDesc:
										switch( value )
										{
											case kPGPike_GR_MODPOne:
											case kPGPike_GR_MODPTwo:
											case kPGPike_GR_MODPFive:
												trx->u.ipsec.groupID =
													(PGPikeGroupID)value;
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_IPSEC_Encapsulation:
										switch( value )
										{
											case kPGPike_PM_Tunnel:
											case kPGPike_PM_Transport:
												break;
											default:
												goto unsupportedTransform;
										}
										break;
									case kPGPike_AT_IPSEC_LifeType:
									{
										PGPikeLifeType	lifeType =
															(PGPikeLifeType) value;
										if( ( ( tep - mp ) >= 4 ) &&
											( ( lifeType == kPGPike_LT_Seconds ) ||
											( lifeType == kPGPike_LT_Kilobytes ) ) )
										{
											pgpIKEGetAttribute( exchange, &mp, tep, &attribute,
																&value, NULL, NULL );
											if( attribute == kPGPike_AT_IPSEC_Duration )
											{
												if( lifeType == kPGPike_LT_Seconds )
													prop->secLifeTime	= value;
												else
													prop->kbLifeTime	= value;
											}
											else
											{
												*alert = kPGPike_AL_InvalidTransform;
												goto done;
											}
										}
										else
										{
											*alert = kPGPike_AL_InvalidTransform;
											goto done;
										}
										break;
									}
									default:
#if PGPIKE_DEBUG
										pgpIKEDebugLog( exchange->ike, TRUE,
											"\tReceived unsupported IPCOMP P2 attribute: %s\n",
											pgpIKEP2AttributeString( attribute ) );
#endif
										*alert = kPGPike_AL_InvalidAttribute;
										goto done;
								}
							}
							prop->numTransforms++;
							break;
						}
						default:
							break;
					}
					if( mp != tep )
					{
						*alert = kPGPike_AL_UnequalPayloadLengths;
						goto done;
					}
				}
				else
				{
					*alert = kPGPike_AL_InvalidTransform;
					goto done;
				}
			}
		unsupportedTransform:
			mp = tep;
		}
		if( ( ( numTransforms > 0 ) && ( prop->numTransforms > 0 ) ) ||
			!numTransforms )
		{
			*proposal = prop;
			prop = NULL;
		}
	}
	else
		err = kPGPError_OutOfMemory;
done:
	if( *alert != kPGPike_AL_None )
		err = pgpIKEAbortExchange( exchangeP, *alert );
	if( IsntNull( prop ) )
		(void)PGPFreeData( prop );
	return err;
}

	void
pgpIKEGetAttribute(
	PGPikeExchange *		exchange,
	PGPByte **				mpp,
	PGPByte *				ep,
	PGPikeAttributeType *	attribute,
	PGPUInt32 *				value,
	PGPByte **				valuePP,
	PGPUInt16 *				valueSize )
{
	PGPByte *				mp = *mpp;
	PGPBoolean				extend;
	PGPUInt16				aVal;
	PGPikeAttributeType		attr;
	
	*attribute = kPGPike_AT_None;
	*value = 0;
	if( IsntNull( valueSize ) )
		*valueSize = 0;
	if( IsntNull( valuePP ) )
		*valuePP = NULL;
	if( ( ep - mp ) < 4 )
		goto done;
	extend = ( *mp & 0x80 ) == 0;
	attr = (PGPikeAttributeType)( PGPEndianToUInt16( kPGPBigEndian, mp ) & 0x7FFF );
	mp	+= sizeof( PGPUInt16 );
	aVal = PGPEndianToUInt16( kPGPBigEndian, mp );
	mp	+= sizeof( PGPUInt16 );
	if( extend )
	{
		PGPBoolean	trueVariable = FALSE;
		
		if( ( ep - mp ) < aVal )
			goto done;
		switch( attr )
		{
			case kPGPike_AT_Config_AppVersion:
			case kPGPike_AT_XAuth_Username:
			case kPGPike_AT_XAuth_Password:
			case kPGPike_AT_XAuth_Passcode:
			case kPGPike_AT_XAuth_Message:
			case kPGPike_AT_XAuth_Challenge:
			case kPGPike_AT_XAuth_Domain:
				if( exchange->exchangeT != kPGPike_EX_Transaction )
					break;
				trueVariable = TRUE;
				break;
			default:
				break;
		}
		if( trueVariable || ( aVal != sizeof(PGPUInt32) ) )
		{
			if( IsntNull( valuePP ) && IsntNull( exchange ) )
			{
				*valuePP = PGPNewSecureData( exchange->ike->memMgr, aVal, 0 );
				if( IsntNull( *valuePP ) )
				{
					pgpCopyMemory( mp, *valuePP, aVal );
					*valueSize = aVal;
				}
			}
			mp += aVal;
		}
		else
		{
			*value = PGPEndianToUInt32( kPGPBigEndian, mp );
			mp += sizeof(PGPUInt32);
		}
	}
	else
		*value = aVal;
	*attribute = attr;
done:
	*mpp = mp;
}

	PGPError
pgpIKEHeaderSniffer(
	PGPikeContextPriv *		ike,
	PGPikeMTPacket *		packet,
	PGPBoolean *			isValid )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp;
	PGPSize					mLen,
							pLen;
	PGPikePayload			payload;
	
	*isValid = FALSE;
	mp		= packet->packet;
	mLen	= packet->packetSize;
	if( mLen < kPGPike_ISAKMPHeaderSize + kPGPike_ISAKMPPayloadHeaderSize )
	{
		err = pgpIKELocalAlert( ike, packet->ipAddress,
				kPGPike_AL_UnequalPayloadLengths, kPGPike_IA_None, FALSE );
		goto done;
	}
	mp += kPGPike_CookieSize * 2;
	/* Payload */
	/* ***** The following could be stricter */
	payload = (PGPikePayload)*mp++;
	switch( payload )
	{
		case kPGPike_PY_SA:
		case kPGPike_PY_KeyExchange:
		case kPGPike_PY_Identification:
		case kPGPike_PY_Certificate:
		case kPGPike_PY_CertRequest:
		case kPGPike_PY_Hash:
		case kPGPike_PY_Signature:
		case kPGPike_PY_Nonce:
		case kPGPike_PY_Notification:
		case kPGPike_PY_Delete:
		case kPGPike_PY_VendorID:
		case kPGPike_PY_Attribute:
			break;
		default:
			/* inappropriate first payload */
			err = pgpIKELocalAlert( ike, packet->ipAddress,
					kPGPike_AL_InvalidPayload, kPGPike_IA_None, FALSE );
			goto done;
	}
	/* Version */
	if( ( *mp >> 4 ) != kPGPike_Version >> 4 )
	{
		err = pgpIKELocalAlert( ike, packet->ipAddress,
				kPGPike_AL_InvalidMajorVersion, kPGPike_IA_None, FALSE );
		goto done;
	}
	if( ( *mp++ & 0x0F ) > ( kPGPike_Version & 0x0F ) )
	{
		err = pgpIKELocalAlert( ike, packet->ipAddress,
				kPGPike_AL_InvalidMinorVersion, kPGPike_IA_None, FALSE );
		goto done;
	}
	/* Exchange */
	switch( *mp++ )
	{
		case kPGPike_EX_Identity:
		case kPGPike_EX_Aggressive:
		case kPGPike_EX_Informational:
		case kPGPike_EX_IPSEC_Quick:
		case kPGPike_EX_Transaction:
			break;
		default:
			err = pgpIKELocalAlert( ike, packet->ipAddress,
					kPGPike_AL_InvalidExchange, kPGPike_IA_None, FALSE );
			goto done;
	}
	mp++;	/* Flags */
	mp += sizeof( PGPikeMessageID );
	pLen = PGPEndianToUInt32( kPGPBigEndian, mp );
	if( pLen > mLen )
	{
		/*	corrupt data */
		
		err = pgpIKELocalAlert( ike, packet->ipAddress,
				kPGPike_AL_UnequalPayloadLengths, kPGPike_IA_None, FALSE );
		goto done;
	}
	mp += sizeof( PGPUInt32 );
	
	*isValid = TRUE;
done:
	return err;
}

	PGPError
pgpIKEPayloadLengthCheck(
	PGPikePartner *			partner,
	PGPikePayload			payload,
	PGPByte *				mp,
	PGPByte *				ep,
	PGPBoolean *			isValid )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePayload			nextPayload;
	PGPUInt16				payLength;
	PGPUInt16				minLen = 0;

	/* Now walk each payload to make sure lengths are all good */
	*isValid = FALSE;
	
#if PGPIKE_DEBUG
	pgpIKEDebugLog( partner->ike, TRUE, "\tPayloads:" );
#endif
	while( payload != kPGPike_PY_None )
	{
		if( ep - mp < kPGPike_ISAKMPPayloadHeaderSize )
			goto done;
		nextPayload = (PGPikePayload)*mp++;
		mp++;	/* reserved */
		payLength = PGPEndianToUInt16( kPGPBigEndian, mp );
		mp += sizeof(PGPUInt16);
		payLength -= kPGPike_ISAKMPPayloadHeaderSize;
		if( ep - mp < payLength )
        {
#if PGPIKE_DEBUG
            pgpIKEDebugLog(partner->ike, FALSE, "\n\tUPL Error: Payload %s, payLen=%d, remainingLen=%d\n", 
                                    pgpIKEPayloadTypeString( payload ), payLength, (ep - mp));
#endif
			goto done;
        }
		switch( payload )
		{
			case kPGPike_PY_SA:
				minLen = 20;
				break;
			case kPGPike_PY_KeyExchange:
			case kPGPike_PY_Identification:
			case kPGPike_PY_Nonce:
			case kPGPike_PY_VendorID:
			case kPGPike_PY_Attribute:
				minLen = 4;
				break;
			case kPGPike_PY_Certificate:
				minLen = 2;
				break;
			case kPGPike_PY_CertRequest:
				minLen = 1;
				break;
			case kPGPike_PY_Hash:
			case kPGPike_PY_Signature:
				minLen = kPGPike_MD5HashSize;
				break;
			case kPGPike_PY_Notification:
				minLen = 8;
				break;
			case kPGPike_PY_Delete:
				minLen = 12;
				break;
			case kPGPike_PY_NATDiscovery:
				minLen = 1;
				break;
			default:
				pgpAssert( 0 );
				goto done;
		}
#if PGPIKE_DEBUG
		pgpIKEDebugLog( partner->ike, FALSE, "%s/", pgpIKEPayloadTypeString( payload ) );
#endif
		if( ep - mp < minLen )
        {
#if PGPIKE_DEBUG
            pgpIKEDebugLog(partner->ike, TRUE, "UPL Error: %s minLen=%d, remainingLen=%d\n", 
                                    pgpIKEPayloadTypeString( payload ), minLen, (ep - mp));
#endif
			goto done;
        }
		payload = nextPayload;
		mp += payLength;
	}
#if PGPIKE_DEBUG
	pgpIKEDebugLog( partner->ike, FALSE, "\n" );
#endif

	*isValid = TRUE;
done:
	return err;
}

	PGPError
pgpIKEInitialContact(
	PGPikeExchange *		exchange )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner	*		partner,
					*		nextPartner;
	PGPipsecSA	*			sa,
			 	*			nextSA;
	
	for( sa = exchange->ike->sa; IsntNull( sa ); )
	{
		nextSA = sa->nextSA;
		if( sa->ipAddress == exchange->partner->ipAddress )
		{
			err = (exchange->ike->msgProc)(
					(PGPikeContextRef)exchange->ike,
					exchange->ike->userData,
					kPGPike_MT_SADied, sa );CKERR;
			err = pgpIKEFreeSA( exchange->ike, sa );	CKERR;
		}
		sa = nextSA;
	}
	for( partner = exchange->ike->partner; IsntNull( partner ); )
	{
		nextPartner = partner->nextPartner;
		if( ( partner->ipAddress == exchange->partner->ipAddress ) &&
			( partner != exchange->partner ) )
		{
			err = pgpIKEFailAllDests( partner, NULL ); CKERR;
			err = pgpIKEFreePartner( partner );	CKERR;
		}
		partner = nextPartner;
	}
#if PGPIKE_DEBUG
	pgpIKEDebugLog( exchange->ike, TRUE, "\t\tInitial Contact Processed\n" );
#endif
done:
	return err;
}

	PGPError
pgpIKEResponderLifetime(
	PGPikeExchange *		exchange,
	PGPByte *				pp,
	PGPByte *				ep )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeAttributeType		attribute,
							ltType,
							ltDuration;
	PGPUInt32				value;

	if( IsntNull( exchange->proposals ) && exchange->initiator )
	{
		if( exchange->exchangeT == kPGPike_EX_Identity )
		{
			ltType		= kPGPike_AT_LifeType;
			ltDuration	= kPGPike_AT_LifeDuration;
		}
		else
		{
			ltType		= kPGPike_AT_IPSEC_LifeType;
			ltDuration	= kPGPike_AT_IPSEC_Duration;
		}
		while( ( ep - pp ) >= 8 )
		{
			PGPikeLifeType	lifeType;
			
			pgpIKEGetAttribute( exchange, &pp, ep, &attribute, &value, NULL, NULL );
			lifeType = (PGPikeLifeType) value;
			if( ( attribute == ltType ) &&
				( ( ep - pp ) >= 4 ) &&
				( ( lifeType == kPGPike_LT_Seconds ) ||
				  ( lifeType == kPGPike_LT_Kilobytes ) ) )
			{
				pgpIKEGetAttribute( exchange, &pp, ep, &attribute, &value, NULL, NULL );
				if( attribute == ltDuration )
				{
					if( lifeType == kPGPike_LT_Seconds )
					{
						if( ( ( exchange->proposals->secLifeTime > value ) ||
								!exchange->proposals->secLifeTime ) &&
							( value >= kPGPike_SecLifeRekeySlack * 2 ) )
						{
							exchange->proposals->secLifeTime	= value;
		#if PGPIKE_DEBUG
							pgpIKEDebugLog( exchange->ike, TRUE,
								"\t\t\tDuration (seconds) reduced: %u\n",
								value );
		#endif
						}
					}
					else
					{
						if( ( ( exchange->proposals->kbLifeTime > value ) ||
								!exchange->proposals->kbLifeTime ) &&
							( value >= kPGPike_KBLifeMinimum ) )
						{
							exchange->proposals->kbLifeTime		= value;
		#if PGPIKE_DEBUG
							pgpIKEDebugLog( exchange->ike, TRUE,
								"\t\t\tDuration (kb) reduced: %u\n",
								value );
		#endif
						}
					}
				}
			}
		}
	}
	return err;
}

	PGPError
pgpIKEFailAllDests(
	PGPikePartner *			partner,
	PGPikeExchange *		onlyExchange )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeMTSAFailed		saf;
	PGPikeExchange *		exchange;
	PGPBoolean				foundDest = FALSE;
	PGPikePendingDest *		pendDest = NULL;
	
	pgpClearMemory( &saf, sizeof( PGPikeMTSAFailed ) );
	saf.ipAddress		= partner->ipAddress;
	for( exchange = partner->exchanges; IsntNull( exchange );
			exchange = exchange->nextExchange )
	{
		if( IsntNull( exchange->destination ) &&
			( IsNull( onlyExchange ) || ( exchange == onlyExchange ) ) )
		{
			foundDest = TRUE;
			saf.ipProtocol	= exchange->destination->ipProtocol;
			saf.ipPort		= exchange->destination->ipPort;
			saf.destIsRange	= exchange->destination->destIsRange;
			saf.ipAddrStart	= exchange->destination->ipAddrStart;
			saf.ipMaskEnd	= exchange->destination->ipMaskEnd;
			err = pgpIKESARequestFailed( exchange->ike, &saf );	CKERR;
			(void)PGPFreeData( exchange->destination );
			exchange->destination = NULL;
		}
	}
	while( IsntNull( partner->pendList ) )
	{
		foundDest = TRUE;
		pendDest = partner->pendList;
		partner->pendList = pendDest->nextDest;
		saf.ipProtocol	= pendDest->ipProtocol;
		saf.ipPort		= pendDest->ipPort;
		saf.destIsRange	= pendDest->destIsRange;
		saf.ipAddrStart	= pendDest->ipAddrStart;
		saf.ipMaskEnd	= pendDest->ipMaskEnd;
		err = pgpIKESARequestFailed( partner->ike, &saf );	CKERR;
		(void)PGPFreeData( pendDest );
	}
	/*if( !foundDest )
		err = pgpIKESARequestFailed( partner->ike, &saf );	CKERR;*/
done:
	return err;
}

	PGPError
pgpIKEProcessInformational(
	PGPikeContextPriv *		ike,
	PGPUInt32				ipAddress,
	PGPikeExchange **		exchangeP,	/* we might delete this exchange */
	PGPikeNDPayload *		nd )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeExchange *		exchange = NULL;
	PGPikeExchange *		p2Exchange;
	PGPikeDOI				doi;
	PGPipsecProtocol		protocol;
	PGPByte					spiSize;
	PGPByte *				pp = nd->pay;
	PGPByte *				ep = nd->pay + nd->payLen;
	
	/* Information exchanges are treated as just that.  Informational.
		Parsing errors in such exchanges are ignored.  For the most part,
		this is due to the fact that informational exchanges are
		largely unprotected except after the initial exchange.  We also
		never reply to an information exchange to avoid potential
		infinite error loops with the remote system.  */
	if( IsntNull( exchangeP ) )
		exchange = *exchangeP;
	if( nd->payLen < 8 )
		goto done;
	doi = (PGPikeDOI)PGPEndianToUInt32( kPGPBigEndian, pp );
	pp += sizeof(PGPikeDOI);
	if( doi != kPGPike_DOI_IPSEC )
		goto done;
	protocol	= (PGPipsecProtocol) *pp++;
	spiSize		= *pp++;
	
	switch( nd->payType )
	{
		case kPGPike_PY_Notification:
		{
			PGPikeAlert			alert;
			
#if PGPIKE_DEBUG
			pgpIKEDebugLog( ike, TRUE, "\tNotification\n" );
#endif
			alert = (PGPikeAlert)PGPEndianToUInt16( kPGPBigEndian, pp );
			pp += sizeof(PGPUInt16);
			if( alert <= kPGPike_AL_LastErrorType )
			{
				err = pgpIKELocalAlert( ike, ipAddress,
						alert, kPGPike_IA_None, TRUE );	CKERR;
			}
			if( IsNull( exchangeP ) )
				break;
			switch( protocol )
			{
				case kPGPike_PR_None:	/* Microsoft sends this, who knows why */
				case kPGPike_PR_AH:
				case kPGPike_PR_ESP:
				case kPGPike_PR_IPCOMP:
				case kPGPike_PR_IKE:
					if( nd->payLen < (PGPSize) (8 + spiSize) )
						goto done;
					pp += spiSize;
					if( IsntNull( exchangeP ) && ( alert <= kPGPike_AL_LastErrorType ) )
					{
						err = pgpIKEFailAllDests( exchange->partner, NULL );	CKERR;
						err = pgpIKEFreePartner( exchange->partner );	CKERR;
						*exchangeP = NULL;	/* the exchange was killed too */
					}
					else switch( alert )
					{
						case kPGPike_AL_InitialContact:
							if( ( exchange->exchangeT == kPGPike_EX_Identity ) ||
								( exchange->exchangeT == kPGPike_EX_Aggressive ) )
								err = pgpIKEInitialContact( exchange );	CKERR;
							break;
						case kPGPike_AL_ResponderLifetime:
#if PGPIKE_DEBUG
							pgpIKEDebugLog( ike, TRUE, "\t\tResponder Lifetime\n" );
#endif
							if( ( exchange->exchangeT == kPGPike_EX_Identity ) ||
								( exchange->exchangeT == kPGPike_EX_Aggressive ) ||
								( exchange->exchangeT == kPGPike_EX_IPSEC_Quick ) )
							{
								err = pgpIKEResponderLifetime( exchange, pp, ep );CKERR;
							}
							break;
						case kPGPike_AL_ReplayStatus:
#if PGPIKE_DEBUG
							pgpIKEDebugLog( exchange->ike, TRUE,
								"\t\tReplay Status - Unsupported\n" );
#endif
							break;
						case kPGPike_AL_Connected:
						{
							for( p2Exchange = exchange->partner->exchanges; IsntNull( p2Exchange );
									p2Exchange = p2Exchange->nextExchange )
							{
								if( p2Exchange->needsCommit &&
									p2Exchange->complete &&
									pgpMemoryEqual( &p2Exchange->messageID,
													&exchange->messageID,
													sizeof(PGPikeMessageID) ) )
								{
									PGPipsecSA *	sa;
									
#if PGPIKE_DEBUG
									pgpIKEDebugLog( exchange->ike, TRUE,
										"\t\tConnected (P2)\n" );
#endif
									err = pgpIKECompletePhase2( p2Exchange, &sa );			CKERR;
									err = pgpIKESAEstablished( p2Exchange->partner, sa );	CKERR;
									err = pgpIKEFreeExchange( p2Exchange );					CKERR;
									break;
								}
							}
							break;
						}
						default:
							break;
					}
					break;
				default:
					break;
			}
			break;
		}
		case kPGPike_PY_Delete:
		{
			PGPUInt16	numSPI,
						spiIndex;
			
			numSPI = PGPEndianToUInt16( kPGPBigEndian, pp );
			pp += sizeof(PGPUInt16);
			if( nd->payLen < 8 + ( (PGPSize) numSPI * spiSize ) )
				goto done;
#if PGPIKE_DEBUG
			pgpIKEDebugLog( exchange->ike, TRUE, "\tDelete (prot=%s)\n",
					( protocol == kPGPike_PR_IKE ) ? "IKE" : "IPSEC" );
#endif
			switch( protocol )
			{
				case kPGPike_PR_IKE:
#if PGPIKE_VERBOSE
					pgpIKEDebugData( exchange->ike, "iCookie", pp,
											sizeof(PGPikeCookie) );
#endif
					if( ( spiSize != kPGPike_CookieSize * 2 ) || ( numSPI != 1 ) )
						goto done;
					if( !pgpMemoryEqual( exchange->partner->initCookie,
											pp, sizeof(PGPikeCookie) ) )
						goto done;
					pp += sizeof(PGPikeCookie);
#if PGPIKE_VERBOSE
					pgpIKEDebugData( exchange->ike, "rCookie", pp,
											sizeof(PGPikeCookie) );
#endif
					if( !pgpMemoryEqual( exchange->partner->respCookie,
											pp, sizeof(PGPikeCookie) ) )
						goto done;
#if PGPIKE_DEBUG
					pgpIKEDebugLog( exchange->ike, TRUE, "\t\tDeleted IKE SA\n" );
#endif
					err = pgpIKEFailAllDests( exchange->partner, NULL );
					err = pgpIKEFreePartner( exchange->partner );	CKERR;
					*exchangeP = NULL;	/* the exchange was killed too */
					break;
				case kPGPike_PR_AH:
				case kPGPike_PR_ESP:
				case kPGPike_PR_IPCOMP:
				{
					PGPipsecSA	*	sa,
								*	nextSA;
					
					if( spiSize != sizeof(PGPipsecSPI) )
						goto done;
					for( spiIndex = 0; spiIndex < numSPI; spiIndex++ )
					{
#if PGPIKE_DEBUG
						pgpIKEDebugData( exchange->ike, "SPI", pp,
											sizeof(PGPipsecSPI) );
#endif
						for( sa = exchange->ike->sa; IsntNull( sa ) && IsntNull( exchange );
								sa = nextSA )
						{
							nextSA = sa->nextSA;
							if( sa->ipAddress == exchange->partner->ipAddress )
							{
								PGPUInt16	trInx;
								PGPBoolean	found = FALSE;
								
								for( trInx = 0; trInx < sa->numTransforms; trInx++ )
								{
									if( pgpMemoryEqual( pp,
											sa->transform[trInx].outSPI,
											sizeof(PGPipsecSPI) ) )
									{
										found = TRUE;
										break;
									}
								}
								if( found )
								{
									PGPikePartner *	partner = exchange->partner;
									
#if PGPIKE_DEBUG
									pgpIKEDebugLog( exchange->ike, TRUE,
										"\t\tDelete IPSEC SA\n" );
#endif
									sa->activeIn = FALSE;
									err = pgpIKEKillSA( &partner, sa ); CKERR;
									if( IsNull( partner ) )
									{
										*exchangeP = NULL;
										exchange = NULL;
									}
								}
							}
						}
						pp += sizeof(PGPipsecSPI);
					}
					break;
				}
				default:
					break;
			}
			break;
		}
		default:
			break;
	}
done:
	return err;
}

	PGPError
pgpIKEKillSA(
	PGPikePartner **		pPartner,
	PGPipsecSA *			sa )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeExchange *		ndExchange = NULL;
	PGPikePartner *			partner = *pPartner;
	PGPikeContextPriv *		ike = partner->ike;
	PGPUInt16				tInx;
	PGPBoolean				deleteIt = FALSE;

	/* send delete exchanges for each protocol's SPIs */
	if( !sa->activeOut || !sa->activeIn )
		deleteIt = TRUE;
	if( sa->activeOut )
		for( tInx = 0; tInx < sa->numTransforms; tInx++ )
		{
			err = pgpIKECreateExchange( partner,
								kPGPike_EX_Informational,
								NULL, kPGPike_S_ND_OneState,
								TRUE, &ndExchange );	CKERR;
			ndExchange->outInfoProtocol = sa->transform[tInx].protocol;
			ndExchange->outInfoSPICount = 1;
			pgpCopyMemory( &sa->transform[tInx].inSPI,
							&ndExchange->outInfoSPI[0],
							sizeof(PGPipsecSPI) );
			err = pgpIKEDoPacket( &ndExchange,
								kPGPike_PY_Delete,
								kPGPike_PY_None );	CKERR;
			(void)pgpIKEFreeExchange( ndExchange );
			ndExchange = NULL;
		}
	sa->activeOut = FALSE;
	if( deleteIt )
	{
		PGPBoolean	found = FALSE;
		PGPipsecSA	*simSA;
		
		/* send SADied message for the SA */
		err = (ike->msgProc)(
				(PGPikeContextRef)ike, ike->userData,
				kPGPike_MT_SADied, sa );CKERR;
		/* make sure there are other SAs that need this partner */
		found = FALSE;
		for( simSA = ike->sa; IsntNull( simSA ); simSA = simSA->nextSA )
		{
			if( ( simSA != sa ) &&
				( simSA->ipAddress == partner->ipAddress ) )
			{
				found = TRUE;
				break;
			}
		}
		if( !found && ( IsNull( partner->exchanges ) ||
			( IsNull( partner->exchanges->nextExchange ) &&
			( partner->exchanges->exchangeT == kPGPike_EX_Informational ) ) ) )
		{
			if( partner->initiator )
			{
				/* no other SAs need this P1 SA, kill that too */
				err = pgpIKECreateExchange( partner,
									kPGPike_EX_Informational,
									NULL, kPGPike_S_ND_OneState,
									TRUE, &ndExchange );	CKERR;
				ndExchange->outInfoProtocol = kPGPike_PR_IKE;
				ndExchange->outInfoSPICount = 0;
#if PGPIKE_DEBUG
				pgpIKEDebugLog( ike, TRUE, "Sending Phase 1 SA Delete (unneeded)\n" );
#endif
				err = pgpIKEDoPacket( &ndExchange,
									kPGPike_PY_Delete,
									kPGPike_PY_None );	CKERR;
				err = pgpIKEFailAllDests( partner, NULL );
				err = pgpIKEFreePartner( partner );	CKERR;
				*pPartner = NULL;
				ndExchange = NULL;
				/* the exchange is deleted automatically */
			}
			else
			{
				/* the other side may have gone down so eventually we may
					need to kill it ourselves */
				partner->termSchedule = PGPGetTime() + kPGPike_SADiedSlack;
			}
		}
	}
	else
	{
		sa->termSchedule = PGPGetTime() + kPGPike_SADiedSlack;
		/* let service know to stop outgoing packets for this SA */
		err = (ike->msgProc)(
				(PGPikeContextRef)ike, ike->userData,
				kPGPike_MT_SAUpdate, sa );CKERR;
	}
	
done:
	if( deleteIt )
		(void)pgpIKEFreeSA( ike, sa );
	if( IsntNull( ndExchange ) )
		(void)pgpIKEFreeExchange( ndExchange );
	return err;
}

	PGPError
pgpIKESARequestFailed(
	PGPikeContextPriv *		ike,
	PGPikeMTSAFailed *		saf )
{
	PGPError				err = kPGPError_NoErr;

	err = pgpIKERemovePending( ike, saf->ipAddress, saf->ipAddrStart,
			saf->ipMaskEnd, saf->destIsRange );	CKERR;
#if PGPIKE_DEBUG
	{
		char	ip1Str[20],
				ip2Str[20],
				ip3Str[20];
		
		pgpIKEGetIPString( saf->ipAddress, ip1Str );
		pgpIKEGetIPString( saf->ipAddrStart, ip2Str );
		pgpIKEGetIPString( saf->ipMaskEnd, ip3Str );
		pgpIKEDebugLog( ike, TRUE, "SAFailed: %s (%s/%s)\n", ip1Str, ip2Str, ip3Str );
	}
#endif
	err = (ike->msgProc)( (PGPikeContextRef)ike, ike->userData,
			kPGPike_MT_SARequestFailed, saf );
done:
	return err;
}

	PGPError
pgpIKEAbortExchange(
	PGPikeExchange **		exchangeP,
	PGPikeAlert				alert )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeExchange *		ndExchange = NULL,
					*		exchange = *exchangeP;
	PGPikeInternalAlert		iAlert = kPGPike_IA_None;
	PGPikePartner	*		oPartner;

	if( alert != kPGPike_AL_None )
	{
		err = pgpIKECreateExchange( exchange->partner,
							kPGPike_EX_Informational,
							NULL, kPGPike_S_ND_OneState,
							TRUE, &ndExchange );	CKERR;
		ndExchange->outAlert			= alert;
		
		if( ( exchange->exchangeT == kPGPike_EX_Identity ) ||
			IsNull( exchange->proposals ) ||
			( (PGPUInt32)exchange->proposals->initSPI == 0 ) ||
			( (PGPUInt32)exchange->proposals->respSPI == 0 ) )
		{
			ndExchange->outInfoProtocol	= kPGPike_PR_IKE;
			ndExchange->outInfoSPICount	= 0;
		}
		else
		{
			ndExchange->outInfoProtocol	= exchange->proposals->protocol;
			ndExchange->outInfoSPICount	= 1;
			if( exchange->initiator )
				pgpCopyMemory( &exchange->proposals->respSPI,
					&ndExchange->outInfoSPI[0], sizeof(PGPipsecSPI) );
			else
				pgpCopyMemory( &exchange->proposals->initSPI,
					&ndExchange->outInfoSPI[0], sizeof(PGPipsecSPI) );
		}
		err = pgpIKEDoPacket( &ndExchange,
							kPGPike_PY_Notification,
							kPGPike_PY_None );	CKERR;
	}
	else
		iAlert = kPGPike_IA_ResponseTimeout;
	err = pgpIKELocalAlert( exchange->ike, exchange->partner->ipAddress,
						alert, iAlert, FALSE );	CKERR;
	err = pgpIKEFailAllDests( exchange->partner, exchange ); CKERR;
	if( !exchange->partner->ready ||
		( ( exchange->partner->authStyle != kPGPike_AS_Normal ) &&
		  !exchange->partner->xAuthenticated ) )
	{
		PGPUInt32			ipAddress = exchange->partner->ipAddress;
		PGPikePartner	*	nextPartner;
		PGPipsecSA		*	sa,
						*	nextSA;
		
		for( sa = exchange->ike->sa; IsntNull( sa ); )
		{
			nextSA = sa->nextSA;
			if( ipAddress == sa->ipAddress )
			{
				err = (exchange->ike->msgProc)(
						(PGPikeContextRef)exchange->ike,
						exchange->ike->userData,
						kPGPike_MT_SADied, sa );CKERR;
				err = pgpIKEFreeSA( exchange->ike, sa );	CKERR;
			}
			sa = nextSA;
		}
		for( oPartner = exchange->ike->partner; IsntNull( oPartner ); )
		{
			nextPartner = oPartner->nextPartner;
			if( ( oPartner != exchange->partner ) &&
				( ipAddress == oPartner->ipAddress ) )
			{
				err = pgpIKEFailAllDests( oPartner, NULL );	CKERR;
				err = pgpIKEFreePartner( oPartner );	CKERR;
			}
			oPartner = nextPartner;
		}
		err = pgpIKEFreePartner( exchange->partner );	CKERR;
		*exchangeP = NULL;
		ndExchange = NULL;
	}
	else
	{
		oPartner = exchange->partner;
		err = pgpIKEFreeExchange( exchange );
		*exchangeP = NULL;
		if( IsNull( oPartner->exchanges ) )
		{
			PGPBoolean		found = FALSE;
			PGPipsecSA *	simSA;
			
			/* If no SAs exist for this P1, schedule it for deletion */
			for( simSA = oPartner->ike->sa; IsntNull( simSA ); simSA = simSA->nextSA )
			{
				if( simSA->ipAddress == oPartner->ipAddress )
				{
					found = TRUE;
					break;
				}
			}
			if( !found )
				oPartner->termSchedule = PGPGetTime();
		}
	}
done:
	if( IsntNull( ndExchange ) )
		(void)pgpIKEFreeExchange( ndExchange );
	if( IsntPGPError( err ) )	/* a big hack to make sure we dont ref the dead */
		err = kPGPError_UserAbort;
	return err;	
}
	
	PGPError
pgpIKELocalAlert(
	PGPikeContextPriv *		ike,
	PGPUInt32				ipAddress,
	PGPikeAlert				alert,
	PGPikeInternalAlert		value,
	PGPBoolean				remote )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeMTAlert			msg;
	
#if PGPIKE_DEBUG
	{
		char	ip1Str[20];
		
		pgpIKEGetIPString( ipAddress, ip1Str );
		pgpIKEDebugLog( ike, TRUE, "ALERT(%c): %s, alert=%s\n",
			remote ? 'R' : 'L', ip1Str, pgpIKEGetAlertString( alert, value ) );
	}
#endif
	msg.ipAddress		= ipAddress;
	msg.alert			= alert;
	msg.value			= value;
	msg.remoteGenerated	= remote;
	err = (ike->msgProc)( (PGPikeContextRef)ike, ike->userData,
							kPGPike_MT_Alert, &msg );
	
	return err;
}

	PGPError
pgpIKEStartIdentityExchange(
	PGPikeContextPriv *		ike,
	PGPUInt32				ipAddress,
	PGPikeMTSASetup *		saSetup )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeMTSASetup			msg;
	PGPBoolean				isRekey = IsNull(saSetup),
							sendXAuthVendorID;
	
	if( IsNull( saSetup ) )
	{
		pgpClearMemory( &msg, sizeof(PGPikeMTSASetup) );
		msg.ipAddress = ipAddress;
		err = (ike->msgProc)( (PGPikeContextRef)ike, ike->userData,
					kPGPike_MT_PolicyCheck, &msg );CKERR;
		saSetup = &msg;
	}
	else
		saSetup->approved = TRUE;
	if( saSetup->approved )
	{
		PGPikePartner	*	partner;
		PGPikeExchange	*	exchange;
		
		err = pgpIKECreatePartner( ike, saSetup, TRUE, &partner );CKERR;
		if( isRekey )
		{
			if( IsntNull( partner->pendList ) )
			{
				(void)PGPFreeData( partner->pendList );
				partner->pendList = NULL;
			}
		}
		err = pgpIKECreateExchange( partner, saSetup->aggressive ?
							kPGPike_EX_Aggressive : kPGPike_EX_Identity, NULL,
							saSetup->aggressive ? kPGPike_S_AM_WaitSA : kPGPike_S_MM_WaitSA,
							TRUE, &exchange );CKERR;
		err = pgpIKECreateProposals( exchange ); CKERR;
		if( IsntNull( exchange->proposals ) &&
			( exchange->proposals->numTransforms == 0 ) )
		{
			err = pgpIKELocalAlert( exchange->ike,
						partner->ipAddress, kPGPike_AL_None,
						kPGPike_IA_NoProposals, FALSE );	CKERR;
			err = pgpIKEFailAllDests( partner, NULL );
			err = pgpIKEFreePartner( partner );
			goto done;
		}
#if PGPIKE_DEBUG
		pgpIKEDebugLog( ike, TRUE, "Initiating Phase 1 Keying\n" );
#endif
		sendXAuthVendorID = ( partner->authStyle != kPGPike_AS_Normal );
		if( saSetup->aggressive )
		{
			PGPBoolean	certReq = FALSE;
			
			if( ( exchange->proposals->t[0].u.ike.authMethod == kPGPike_AM_DSS_Sig ) ||
						( exchange->proposals->t[0].u.ike.authMethod == kPGPike_AM_RSA_Sig ) )
				certReq = TRUE;
			err = pgpIKELoadGroup( exchange );	CKERR;
			err = pgpIKEDoPacket( &exchange,
						kPGPike_PY_SA,
						kPGPike_PY_KeyExchange,
						kPGPike_PY_Nonce,
						kPGPike_PY_Identification,
						certReq ? kPGPike_PY_CertRequest : kPGPike_PY_Skip,
						kPGPike_PY_VendorID,
						kPGPike_PY_VendorID,
						sendXAuthVendorID ? kPGPike_PY_VendorID : kPGPike_PY_Skip,
						kPGPike_PY_None );		CKERR;
		}
		else
		{
            err = pgpIKEDoPacket( &exchange,
						kPGPike_PY_SA,
						kPGPike_PY_VendorID,
						kPGPike_PY_VendorID,
						sendXAuthVendorID ? kPGPike_PY_VendorID : kPGPike_PY_Skip,
						kPGPike_PY_None );		CKERR;
		}
		if( saSetup->lightweight )
			exchange->retry = 1;
	}
done:
	return err;
}

	PGPError
pgpIKEStartConfigRequest(
	PGPikePartner *		partner,
	PGPikeExchange **	exchangeP )
{
	PGPError			err = kPGPError_NoErr;
	PGPikeExchange *	exchange = NULL;
	
#if PGPIKE_DEBUG
	pgpIKEDebugLog( partner->ike, TRUE, "Initiating Config Request\n" );
#endif
	if( IsntNull( exchangeP ) )
		*exchangeP = NULL;
	err = pgpIKECreateExchange( partner,
					kPGPike_EX_Transaction, NULL,
					kPGPike_S_ND_OneState,
					TRUE, &exchange );	CKERR;
	
	/* We have a fixed set of Requests that we make of the gateway. */
	exchange->numConfigs = 4;
	exchange->configs[0].configType		= kPGPike_CT_Request;
	exchange->configs[0].transactionID	= 0;
	exchange->configs[0].attribute		= kPGPike_AT_Config_IPv4Address;
	exchange->configs[0].value			= 0;
	exchange->configs[1].configType		= kPGPike_CT_Request;
	exchange->configs[1].transactionID	= 0;
	exchange->configs[1].attribute		= kPGPike_AT_Config_IPv4DNS;
	exchange->configs[1].value			= 0;
	exchange->configs[2].configType		= kPGPike_CT_Request;
	exchange->configs[2].transactionID	= 0;
	exchange->configs[2].attribute		= kPGPike_AT_Config_IPv4NBNS;
	exchange->configs[2].value			= 0;
	exchange->configs[3].configType		= kPGPike_CT_Request;
	exchange->configs[3].transactionID	= 0;
	exchange->configs[3].attribute		= kPGPike_AT_Config_AddressExpiry;
	exchange->configs[3].value			= 0;
	
	err = pgpIKEDoPacket( &exchange,
				kPGPike_PY_Attribute,
				kPGPike_PY_None );	CKERR;
	if( IsntNull( exchangeP ) )
		*exchangeP = exchange;
done:
	return err;
}

	PGPError
pgpIKEStartQMExchange(
	PGPikePartner *		partner,
	PGPikeExchange **	exchangeP )
{
	PGPError			err = kPGPError_NoErr;
	PGPikeExchange *	exchange = NULL;
	PGPBoolean			pfs = FALSE,
						idC = FALSE;
	
#if PGPIKE_DEBUG
	pgpIKEDebugLog( partner->ike, TRUE, "Initiating Phase 2 QM\n" );
#endif
	if( IsntNull( exchangeP ) )
		*exchangeP = NULL;
	err = pgpIKECreateExchange( partner,
					kPGPike_EX_IPSEC_Quick, NULL,
					kPGPike_S_QM_WaitSA,
					TRUE, &exchange );	CKERR;
	pgpAssert( IsntNull( partner->pendList ) );
	exchange->destination				= partner->pendList;
	partner->pendList					= partner->pendList->nextDest;
	exchange->destination->nextDest		= NULL;
	
	err = pgpIKECreateProposals( exchange );	CKERR;
	if( exchange->proposals->t[0].u.ipsec.groupID !=
		kPGPike_GR_None )
	{
		pfs = TRUE;
		err = pgpIKELoadGroup( exchange );	CKERR;
	}
	if( exchange->destination->ipAddrStart )
		idC = TRUE;
	
	err = pgpIKEDoPacket( &exchange,
				kPGPike_PY_SA,
				kPGPike_PY_Nonce,
				pfs ? kPGPike_PY_KeyExchange : kPGPike_PY_Skip,
				idC ? kPGPike_PY_Identification : kPGPike_PY_Skip,
				idC ? kPGPike_PY_Identification : kPGPike_PY_Skip,
				kPGPike_PY_None );	CKERR;
	if( IsntNull( exchangeP ) )
		*exchangeP = exchange;
done:
	return err;
}

	PGPError
pgpIKEIdle(
	PGPikeContextPriv *		ike )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner	*		partner,
					*		nextPartner;
	PGPikeExchange	*		exchange;
	PGPipsecSA		*		sa,
					*		nextSA;
	PGPUInt32				curMillisec;
	PGPTime					curTime,
							deathTime;
	PGPBoolean				found;

	curMillisec	= PGPGetMilliseconds();
	curTime		= PGPGetTime();
	for( partner = ike->partner; IsntNull( partner ); partner = nextPartner )
	{
		nextPartner = partner->nextPartner; /* save in case partner dies */
		
		/* Cycle through all exchanges to check for retransmittable packets */
		for( exchange = partner->exchanges; IsntNull( exchange );
				exchange = exchange->nextExchange )
		{
			PGPUInt32		retxTime;
			
			if( !exchange->lastTransmit )
				continue;
			if( exchange->complete )
				retxTime = kPGPike_CommitBitSlack;
			else
				retxTime = partner->rttMillisec + kPGPike_RoundTripSlack;
			if( curMillisec - exchange->lastTransmit > retxTime )
			{
				if( !partner->rttMillisec )
					partner->rttMillisec = kPGPike_RoundTripSlack;
				else
				{
					partner->rttMillisec *= 2;
					if( partner->rttMillisec > kPGPike_MaxRoundTrip )
						partner->rttMillisec = kPGPike_MaxRoundTrip;
				}
				err = pgpIKEResendLastPacket( exchange );	CKERR;
			}
		}
		
		if( partner->ready && IsntNull( partner->pendList ) &&
			IsNull( partner->exchanges ) &&
			( partner->xAuthenticated || ( partner->authStyle == kPGPike_AS_Normal ) ) )
		{
			err = pgpIKEStartQMExchange( partner, NULL );	CKERR;
		}
		else if( partner->assignedExpiry && partner->ready && IsNull( partner->exchanges ) )
		{
			if( ( partner->assignedExpiry <= curTime ) && partner->virtualIP )
			{
				err = pgpIKEStartConfigRequest( partner, NULL );	CKERR;
			}
		}
		if(	partner->ready && ( partner->secLifeTime || partner->termSchedule ) &&
			IsNull( partner->exchanges ) )
		{
			/* kPGPike_SecLifeRespSlack makes responders auto-delete P1 SAs
				slightly slower than initiators to avoid unpleasant alerts
				when both sides delete simultaneously */
			
			deathTime = ( ( partner->termSchedule > 0 ) ?
						partner->termSchedule :
						( partner->birthTime + partner->secLifeTime ) ) +
						( partner->initiator ? 0 : kPGPike_SecLifeRespSlack );
			if( curTime >= deathTime )
			{
				/* partner's dead Jim */
				
				err = pgpIKECreateExchange( partner,
									kPGPike_EX_Informational,
									NULL, kPGPike_S_ND_OneState,
									TRUE, &exchange );	CKERR;
				exchange->outInfoProtocol = kPGPike_PR_IKE;
				exchange->outInfoSPICount = 0;
#if PGPIKE_DEBUG
				pgpIKEDebugLog( ike, TRUE, "Sending Phase 1 SA Delete (expired)\n" );
#endif
				err = pgpIKEDoPacket( &exchange,
									kPGPike_PY_Delete,
									kPGPike_PY_None );	CKERR;
				err = pgpIKEFailAllDests( partner, NULL );	CKERR;
				err = pgpIKEFreePartner( partner );	CKERR;
				/* the exchange is deleted automatically */
			}
			else if( partner->initiator &&
					( deathTime - curTime <= kPGPike_SecLifeRekeySlack ) )
			{
				PGPikePartner *		oPartner;
				
				/* try to rekey the IKE SA if there are no others */
				
				found = FALSE;
				for( oPartner = ike->partner; IsntNull( oPartner );
						oPartner = oPartner->nextPartner )
				{
					if( ( oPartner != partner ) &&
						( oPartner->ipAddress == partner->ipAddress ) )
					{
						found = TRUE;
						break;
					}
				}
				if( !found )
				{
					/* and make sure a P2 SA exists that actually would want us to
						rekey this P1 SA */
					found = FALSE;
					for( sa = ike->sa; IsntNull( sa ); sa = sa->nextSA )
					{
						if( sa->ipAddress == partner->ipAddress )
						{
							found = TRUE;
							break;
						}
					}
					if( found )
					{
						err = pgpIKEStartIdentityExchange( ike, partner->ipAddress, NULL );	CKERR;
					}
				}
			}
		}
	}
	
	/* Cycle through all SAs to check for dead or dying */
	for( sa = ike->sa; IsntNull( sa ); sa = nextSA )
	{
		nextSA = sa->nextSA;
		
		if( sa->termSchedule )
		{
			if( sa->termSchedule < curTime )
			{
				if( pgpIKEFindSAPartner( ike, sa, TRUE, &partner ) )
				{
					err = pgpIKEKillSA( &partner, sa );	CKERR;
				}
				else
				{
					err = (ike->msgProc)(
							(PGPikeContextRef)ike,
							ike->userData, kPGPike_MT_SADied, sa );CKERR;
					err = pgpIKEFreeSA( ike, sa );	CKERR;
				}
			}
		}
		else if( sa->secLifeTime )
		{
			deathTime = sa->birthTime + sa->secLifeTime;
			if( curTime >= deathTime )
			{
				/* he's dead Jim */
				
				if( pgpIKEFindSAPartner( ike, sa, TRUE, &partner ) )
				{
					err = pgpIKEKillSA( &partner, sa );	CKERR;
				}
				else
				{
					/* no partner exists, we'll have to negotiate one */
					if( !pgpIKEFindSAPartner( ike, sa, FALSE, &partner ) )
					{
#if PGPIKE_DEBUG
						pgpIKEDebugLog( ike, TRUE, "No partners for SA Delete: Creating\n" );
#endif
						sa->termSchedule = curTime + kPGPike_NoPartnerSlack;
						err = pgpIKEStartIdentityExchange( ike, sa->ipAddress, NULL );	CKERR;
					}
				}
			}
			else if( deathTime - curTime <= kPGPike_SecLifeRekeySlack )
			{
				PGPipsecSA	*	oSA;

				found = FALSE;
				for( oSA = ike->sa; IsntNull( oSA ); oSA = oSA->nextSA )
				{
					if( ( sa != oSA ) && ( oSA->ipAddress == sa->ipAddress ) &&
						( oSA->destIsRange == sa->destIsRange ) &&
						( oSA->ipAddrStart == sa->ipAddrStart ) &&
						( oSA->ipMaskEnd == sa->ipMaskEnd ) )
					{
						found = TRUE;
						break;
					}
				}
				if( !found &&
					pgpIKEFindSAPartner( ike, sa, TRUE, &partner ) &&
					partner->initiator && IsNull( partner->exchanges ) )
				{
					PGPikePendingDest *	rekeyDest;
					/* we were the initiator and should rekey this SA */
					
#if PGPIKE_DEBUG
					pgpIKEDebugLog( ike, TRUE, "Initiating Phase 2 Rekey\n" );
#endif
					rekeyDest = (PGPikePendingDest *)PGPNewData( ike->memMgr,
									sizeof(PGPikePendingDest), kPGPMemoryMgrFlags_Clear );
					if( IsNull( rekeyDest ) )
					{
						err = kPGPError_OutOfMemory;
						goto done;
					}
					rekeyDest->ipAddrStart	= sa->ipAddrStart;
					rekeyDest->ipMaskEnd	= sa->ipMaskEnd;
					rekeyDest->destIsRange	= sa->destIsRange;
					rekeyDest->ipPort		= sa->ipPort;
					rekeyDest->ipProtocol	= sa->ipProtocol;
					rekeyDest->nextDest		= partner->pendList;
					partner->pendList		= rekeyDest;
					err = pgpIKEStartQMExchange( partner, &exchange );	CKERR;
				}
			}
		}
	}
done:
	return err;	
}

/* The Mighty IKE Packet Construction Engine, next 100 exits */
	PGPError
pgpIKEDoPacket(
	PGPikeExchange **		exchangeP,
	PGPikePayload			firstPayload,
	... )
{
	va_list					morePayloads;
	PGPError				err = kPGPError_NoErr;
	PGPikeExchange *		exchange = *exchangeP;
	PGPikePayload			payload;
	PGPByte *				packet = NULL;
	PGPByte *				mp;
	PGPByte *				hp;
	PGPSize					packetSize = kPGPike_ISAKMPHeaderSize,
							allocSize = kPGPike_DefaultPacketAlloc;
	PGPBoolean				commitBit	= FALSE,
							authOnlyBit	= FALSE,
							retry		= TRUE,
							encrypt		= FALSE;
	PGPHMACContextRef		p2hmac = kInvalidPGPHMACContextRef;
	
	va_start( morePayloads, firstPayload );

	pgpAssertAddrValid( exchange, PGPikeExchange * );
	
	mp = packet = PGPNewData( exchange->ike->memMgr, allocSize,
						kPGPMemoryMgrFlags_Clear );
	if( IsNull( packet ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}
	if( PGPCBCContextRefIsValid( exchange->partner->cbc ) &&
		( ( exchange->exchangeT == kPGPike_EX_Identity ) ||
		  ( exchange->exchangeT == kPGPike_EX_IPSEC_Quick ) ||
		  ( exchange->exchangeT == kPGPike_EX_Aggressive ) ||
		( ( ( exchange->exchangeT == kPGPike_EX_Informational ) ||
			( exchange->exchangeT == kPGPike_EX_Transaction ) ) &&
			exchange->partner->ready ) ) )
		encrypt = TRUE;
#if PGPIKE_DEBUG
	pgpIKEDebugLog( exchange->ike, TRUE, "Send: %s", encrypt ? "(E):" : "" );
#endif
	/* Payloads */
	mp = packet + kPGPike_ISAKMPHeaderSize;
	if( encrypt && ( exchange->exchangeT != kPGPike_EX_Identity ) &&
		 ( exchange->exchangeT != kPGPike_EX_Aggressive ) )
	{
		hp = mp += kPGPike_ISAKMPPayloadHeaderSize + exchange->partner->hashSize;
		err = PGPNewHMACContext( exchange->ike->pgpContext,
									exchange->partner->sdkHashAlg,
									exchange->partner->skeyid_a,
									exchange->partner->hashSize,
									&p2hmac );	CKERR;
		if( exchange->initiator && exchange->state == kPGPike_S_QM_WaitHash3 )
		{
			err = PGPContinueHMAC( p2hmac, "\0", 1 );	CKERR;
		}
		err = PGPContinueHMAC( p2hmac, exchange->messageID,
								sizeof(PGPikeMessageID) );	CKERR;
		if( exchange->state == kPGPike_S_QM_WaitHash3 )
		{
			err = PGPContinueHMAC( p2hmac, exchange->initNonce,
											exchange->initNonceLen );CKERR;
			if( exchange->initiator )
			{
				err = PGPContinueHMAC( p2hmac, exchange->respNonce,
											exchange->respNonceLen );CKERR;
			}
		}
#if PGPIKE_DEBUG
		if( exchange->exchangeT != kPGPike_EX_Identity )
			pgpIKEDebugLog( exchange->ike, FALSE, "%s/",
				pgpIKEPayloadTypeString( kPGPike_PY_Hash ) );
#endif
	}
	for( payload = firstPayload; payload != kPGPike_PY_None; )
	{
		PGPByte *	hdr = mp;
		PGPUInt16	plen;
		
		mp += kPGPike_ISAKMPPayloadHeaderSize;
#if PGPIKE_DEBUG
		pgpIKEDebugLog( exchange->ike, FALSE, "%s/", pgpIKEPayloadTypeString( payload ) );
#endif
		switch( payload )
		{
			case kPGPike_PY_SA:
				err = pgpIKEAddSAPayload( exchange, &mp ); CKERR;
				if( ( exchange->state == kPGPike_S_MM_WaitSA ) ||
					  ( exchange->state == kPGPike_S_AM_WaitSA ) )
				{
					if( exchange->initiator )
					{
						/* save the SA payload body for HASH_X */
						pgpAssert( IsNull( exchange->initSABody ) );
						exchange->initSABodySize = mp - hdr - kPGPike_ISAKMPPayloadHeaderSize;
						exchange->initSABody = PGPNewData( exchange->ike->memMgr,
										exchange->initSABodySize,
										kPGPMemoryMgrFlags_Clear );
						if( IsNull( exchange->initSABody ) )
						{
							err = kPGPError_OutOfMemory;
							goto done;
						}
						pgpCopyMemory( hdr + kPGPike_ISAKMPPayloadHeaderSize,
										exchange->initSABody,
										exchange->initSABodySize );
					}
					else
					{
						switch( exchange->proposals->t[0].u.ike.authMethod )
						{
							case kPGPike_AM_XAuth_RespPreShared:
								exchange->proposals->t[0].u.ike.authMethod = kPGPike_AM_PreSharedKey;
								break;
							case kPGPike_AM_XAuth_RespDSS:
							case kPGPike_AM_HAuth_RespDSS:
								exchange->proposals->t[0].u.ike.authMethod = kPGPike_AM_DSS_Sig;
								break;
							case kPGPike_AM_XAuth_RespRSA:
							case kPGPike_AM_HAuth_RespRSA:
								exchange->proposals->t[0].u.ike.authMethod = kPGPike_AM_RSA_Sig;
								break;
							default:
								break;
						}
					}
				}
				break;
			case kPGPike_PY_KeyExchange:
				err = pgpIKEAddKEPayload( exchange, &mp ); CKERR;
				break;
			case kPGPike_PY_Identification:
				err = pgpIKEAddIDPayload( exchange, &mp ); CKERR;
				break;
			case kPGPike_PY_Certificate:
				err = pgpIKEAddCertPayload( exchange, &mp ); CKERR;
				break;
			case kPGPike_PY_CertRequest:
				err = pgpIKEAddCertRequestPayload( exchange, &mp ); CKERR;
				break;
			case kPGPike_PY_Signature:
				err = pgpIKEAddSigPayload( exchange, &mp ); CKERR;
				break;
			case kPGPike_PY_Nonce:
				err = pgpIKEAddNoncePayload( exchange, &mp ); CKERR;
				break;
			case kPGPike_PY_Hash:
				/* used only for main/aggr mode, QM Hashes are outside this loop */
				err = pgpIKEGetAuthHash( exchange, exchange->initiator, mp );
				mp += exchange->partner->hashSize;
				break;
			case kPGPike_PY_Notification:
			{
				PGPUInt32ToEndian( (PGPUInt32) exchange->doi,
						kPGPBigEndian, mp );
				mp += sizeof(PGPUInt32);
				*mp++ = exchange->outInfoProtocol;
				if( exchange->outInfoProtocol == kPGPike_PR_IKE )
				{
					*mp++ = 0;
					PGPUInt16ToEndian( (PGPUInt16) exchange->outAlert,
							kPGPBigEndian, mp );
					mp += sizeof(PGPUInt16);
					if( exchange->outAlert == kPGPike_AL_ResponderLifetime )
					{
						if( exchange->outRLSeconds )
						{
							pgpIKEAddAttribute( kPGPike_AT_LifeType,
								kPGPike_LT_Seconds, NULL, 0, &mp );
							pgpIKEAddAttribute( kPGPike_AT_LifeDuration,
								exchange->ike->secLifeTimeIKE, NULL, 0, &mp );
							exchange->proposals->secLifeTime = exchange->ike->secLifeTimeIKE;
						}
						if( exchange->outRLKB )
						{
							pgpIKEAddAttribute( kPGPike_AT_LifeType,
								kPGPike_LT_Kilobytes, NULL, 0, &mp );
							pgpIKEAddAttribute( kPGPike_AT_LifeDuration,
								exchange->ike->kbLifeTimeIKE, NULL, 0, &mp );
							exchange->proposals->kbLifeTime = exchange->ike->kbLifeTimeIKE;
						}
					}
				}
				else
				{
					*mp++ = sizeof(PGPipsecSPI);
					PGPUInt16ToEndian( (PGPUInt16) exchange->outAlert,
							kPGPBigEndian, mp );
					mp += sizeof(PGPUInt16);
					pgpAssert( exchange->outInfoSPICount == 1 );
					pgpCopyMemory( &exchange->outInfoSPI[0], mp,
									sizeof(PGPipsecSPI) );
					mp += sizeof(PGPipsecSPI);
					if( exchange->outAlert == kPGPike_AL_ResponderLifetime )
					{
						if( exchange->outRLSeconds )
						{
							pgpIKEAddAttribute( kPGPike_AT_IPSEC_LifeType,
								kPGPike_LT_Seconds, NULL, 0, &mp );
							pgpIKEAddAttribute( kPGPike_AT_IPSEC_Duration,
								exchange->ike->secLifeTimeIPSEC, NULL, 0, &mp );
							exchange->proposals->secLifeTime = exchange->ike->secLifeTimeIPSEC;
						}
						if( exchange->outRLKB )
						{
							pgpIKEAddAttribute( kPGPike_AT_IPSEC_LifeType,
								kPGPike_LT_Kilobytes, NULL, 0, &mp );
							pgpIKEAddAttribute( kPGPike_AT_IPSEC_Duration,
								exchange->ike->kbLifeTimeIPSEC, NULL, 0, &mp );
							exchange->proposals->kbLifeTime = exchange->ike->kbLifeTimeIPSEC;
						}
					}
				}
				break;
			}
			case kPGPike_PY_Delete:
				PGPUInt32ToEndian( (PGPUInt32) exchange->doi,
						kPGPBigEndian, mp );
				mp += sizeof(PGPUInt32);
				*mp++ = exchange->outInfoProtocol;
				if( exchange->outInfoProtocol == kPGPike_PR_IKE )
				{
					*mp++ = sizeof(PGPikeCookie) * 2;
					*mp++ = 0;
					*mp++ = 1;
					pgpCopyMemory( exchange->partner->initCookie, mp,
									sizeof(PGPikeCookie) );
					mp += sizeof(PGPikeCookie);
					pgpCopyMemory( exchange->partner->respCookie, mp,
									sizeof(PGPikeCookie) );
					mp += sizeof(PGPikeCookie);
				}
				else
				{
					PGPUInt16	sInx;
					
					*mp++ = sizeof(PGPipsecSPI);
					PGPUInt16ToEndian( exchange->outInfoSPICount,
							kPGPBigEndian, mp );
					mp += sizeof(PGPUInt16);
					for( sInx = 0; sInx < exchange->outInfoSPICount; sInx++ )
					{
						pgpCopyMemory( &exchange->outInfoSPI[sInx], mp,
										sizeof(PGPipsecSPI) );
						mp += sizeof(PGPipsecSPI);
					}
				}
				break;
			case kPGPike_PY_Attribute:
				err = pgpIKEAddAttrPayload( exchange, &mp ); CKERR;
				break;
			case kPGPike_PY_NATDiscovery:
				err = pgpIKEAddNATDiscoveryPayload( exchange, &mp ); CKERR;
				break;
			case kPGPike_PY_VendorID:
			{
				PGPUInt16	vLen;
				
				if( !exchange->sentPGPVendorID )
				{
					vLen = strlen(kPGPike_PGPVendorString1);
					pgpCopyMemory( kPGPike_PGPVendorString1, mp, vLen );
					mp += vLen;
					vLen = strlen(kPGPike_PGPVendorString2);
					pgpCopyMemory( kPGPike_PGPVendorString2, mp, vLen );
					mp += vLen;
					exchange->sentPGPVendorID = TRUE;
				}
				else if( !exchange->sentNATTravVendorID )
				{
					vLen = sizeof(kPGPike_NATTraversalVendorID);
					pgpCopyMemory(kPGPike_NATTraversalVendorID, mp, vLen);
					mp += vLen;
					exchange->sentNATTravVendorID = TRUE;
				}
				else
				{
					pgpCopyMemory( kPGPike_XAuth6VendorID, mp, sizeof(kPGPike_XAuth6VendorID) );
					mp += sizeof(kPGPike_XAuth6VendorID);
				}
				break;
			}
			case kPGPike_PY_Proposal:
			case kPGPike_PY_Transform:
			default:
				/* inappropriate payload requested */
				pgpAssert( 0 );
				break;
		}
		pgpAssert( packetSize < kPGPike_DefaultPacketAlloc );
		
		do
		{
			payload = va_arg( morePayloads, PGPikePayload );
		} while( payload == kPGPike_PY_Skip );

		plen = mp - hdr;
		*hdr++ = payload;
		*hdr++ = 0; /* reserved */
		PGPUInt16ToEndian( plen, kPGPBigEndian, hdr );
	}
	if( PGPHMACContextRefIsValid( p2hmac ) )
	{
		err = PGPContinueHMAC( p2hmac, hp, mp - hp );	CKERR;
		hp -= exchange->partner->hashSize;
		err = PGPFinalizeHMAC( p2hmac, hp );	CKERR;
		hp -= kPGPike_ISAKMPPayloadHeaderSize;
		*hp++ = firstPayload;
		firstPayload = kPGPike_PY_Hash;
		*hp++ = 0;	/* reserved */
		PGPUInt16ToEndian(
			(PGPUInt16)( exchange->partner->hashSize +
			kPGPike_ISAKMPPayloadHeaderSize ),
			kPGPBigEndian, hp );
	}
	packetSize = mp - packet;
	if( encrypt )
	{
		/* Pad the packet out for the CBC encryption */
		while( ( packetSize - kPGPike_ISAKMPHeaderSize ) %
				kPGPike_MaxCipherBlockSize )
		{
			*mp++ = 0;
			packetSize++;
		}
		/* Encrypt the data */
		err = PGPInitCBC( exchange->partner->cbc,
							exchange->partner->cipherKey,
							exchange->lastCBC );	CKERR;
		err = PGPCBCEncrypt( exchange->partner->cbc,
								packet + kPGPike_ISAKMPHeaderSize,
								packetSize - kPGPike_ISAKMPHeaderSize,
								packet + kPGPike_ISAKMPHeaderSize );	CKERR;
		/* Save the IV for the next packet */
		pgpCopyMemory( packet + packetSize - kPGPike_MaxCipherBlockSize,
						exchange->lastCBC,
						kPGPike_MaxCipherBlockSize );
		if( ( exchange->exchangeT == kPGPike_EX_Identity ) ||
			( exchange->exchangeT == kPGPike_EX_Aggressive ) )
			pgpCopyMemory( exchange->lastCBC, exchange->partner->lastP1CBC,
						kPGPike_MaxCipherBlockSize );
	}
	/* ISAKMP Header */
	mp = packet;
	pgpCopyMemory( exchange->partner->initCookie, mp, kPGPike_CookieSize );
	mp += kPGPike_CookieSize;
	pgpCopyMemory( exchange->partner->respCookie, mp, kPGPike_CookieSize );
	mp += kPGPike_CookieSize;
	*mp++ = firstPayload;
	*mp++ = kPGPike_Version;
	*mp++ = exchange->exchangeT;
	*mp = 0;	/* Flags */
	if( encrypt )
		*mp |= kPGPike_ISAKMPEncryptBit;
	if( commitBit )
		*mp |= kPGPike_ISAKMPCommitBit;
	if( authOnlyBit )
		*mp |= kPGPike_ISAKMPAuthOnlyBit;
	mp++;
	pgpCopyMemory( exchange->messageID, mp, kPGPike_MessageIDSize );
	mp+= kPGPike_MessageIDSize;
	PGPUInt32ToEndian( (PGPUInt32) packetSize, kPGPBigEndian, mp );
	
	/* resize to its natural size */
	err = PGPReallocData( exchange->ike->memMgr, (void **)&packet, packetSize, 0 );CKERR;
	/* done, send the packet */
	if( retry )
	{
		err = pgpIKESendPacket( exchange, packet, packetSize );	CKERR;
	}
	else
	{
		err = pgpIKESendPacket1( exchange->partner, packet, packetSize ); CKERR;
		err = PGPFreeData( packet );
	}
	packet = NULL;
done:
#if PGPIKE_DEBUG
	pgpIKEDebugLog( exchange->ike, FALSE, "\n" );
#endif
	if( IsPGPError( err ) )
	{
		/* Not clear what error to send out here.  Any PGPError in this
			function indicates a bug in pgpIKE or the calling application.
			We fudge an error here for safety. */
		pgpIKEAbortExchange( exchangeP, kPGPike_AL_UnsupportedExchange );
	}
	if( IsntNull( packet ) )
		PGPFreeData( packet );
	if( PGPHMACContextRefIsValid( p2hmac ) )
		(void)PGPFreeHMACContext( p2hmac );

	va_end( morePayloads );
	return err;
}

	PGPError
pgpIKEAddIDPayload(
	PGPikeExchange *		exchange,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	PGPByte **				pBody;
	PGPSize *				pSize;
	
	if( exchange->exchangeT == kPGPike_EX_IPSEC_Quick )
	{
		PGPBoolean	idCi = TRUE;
		
		pBody = &exchange->idBody;
		pSize = &exchange->idBodySize;
		if( exchange->initiator )
		{
			if( IsntNull( exchange->idBody ) )
			{
				pBody = &exchange->idRBody;
				pSize = &exchange->idRBodySize;
				idCi = FALSE;
			}
		}
		else
		{
			if( IsNull( exchange->idBody ) )
				idCi = FALSE;
			else
			{
				pBody = &exchange->idRBody;
				pSize = &exchange->idRBodySize;
			}
		}
		if( idCi )
		{
			*mp++ = kPGPike_ID_IPV4_Addr;											/* ID type */
			*mp++ = exchange->destination->ipProtocol;								/* protocol ID */
			PGPUInt16ToEndian( exchange->destination->ipPort, kPGPBigEndian, mp );	/* port */
			mp += sizeof(PGPUInt16);
			if( exchange->partner->assignedIP )
				pgpCopyMemory( &exchange->partner->assignedIP, mp, sizeof(PGPUInt32) );
			else
				pgpCopyMemory( &exchange->partner->localIPAddress, mp, sizeof(PGPUInt32) );
			mp += sizeof(PGPUInt32);
		}
		else
		{
			PGPipsecIdentity	idType;
			
			/* BUG?***** */
			if( exchange->destination->destIsRange )
				idType	= kPGPike_ID_IPV4_Addr_Range;
			else if( exchange->destination->ipMaskEnd != 0xFFFFFFFF )
				idType	= kPGPike_ID_IPV4_Addr_Subnet;
			else
				idType	= kPGPike_ID_IPV4_Addr;
			*mp++ = idType;															/* ID type */
			*mp++ = exchange->destination->ipProtocol;								/* protocol ID */
			PGPUInt16ToEndian( exchange->destination->ipPort, kPGPBigEndian, mp );	/* port */
			mp += sizeof(PGPUInt16);
			switch( idType )
			{
				case kPGPike_ID_IPV4_Addr:
					pgpCopyMemory( &exchange->destination->ipAddrStart, mp, sizeof(PGPUInt32) );
					break;
				case kPGPike_ID_IPV4_Addr_Subnet:
				case kPGPike_ID_IPV4_Addr_Range:
					if( ( exchange->destination->ipAddrStart == 1 ) &&
						( exchange->destination->ipMaskEnd == 0 ) )	/* exc gateway special case */
						PGPUInt32ToEndian( 0, kPGPBigEndian, mp );
					else
						pgpCopyMemory( &exchange->destination->ipAddrStart, mp, sizeof(PGPUInt32) );
					mp += sizeof(PGPUInt32);
					pgpCopyMemory( &exchange->destination->ipMaskEnd, mp, sizeof(PGPUInt32) );
					break;
				default:
					break;
			}
			mp += sizeof(PGPUInt32);
		}
	}
	else
	{
		PGPikeAuthMethod	authMethod;
		
		if( exchange->initiator )
		{
			pBody = &exchange->idBody;
			pSize = &exchange->idBodySize;
		}
		else
		{
			pBody = &exchange->idRBody;
			pSize = &exchange->idRBodySize;
		}
		if( exchange->partner->authStyle == kPGPike_AS_HybridAuth )
			authMethod = kPGPike_AM_PreSharedKey;
		else
			authMethod = exchange->proposals->t[0].u.ike.authMethod;
		switch( authMethod )
		{
			case kPGPike_AM_DSS_Sig:
			case kPGPike_AM_RSA_Sig:
			{
				PGPKeyDBObjRef		pgpKey;
				PGPKeyDBObjRef		x509Cert;
				
				pgpKey = exchange->partner->pgpAuthKey.authObj;
				x509Cert = exchange->partner->x509AuthKey.authObj;
				if( ( exchange->partner->remotePGPVersion && PGPKeyDBObjRefIsValid( pgpKey ) ) ||
					( ( exchange->exchangeT == kPGPike_EX_Aggressive ) &&
						PGPKeyDBObjRefIsValid( pgpKey ) && !PGPKeyDBObjRefIsValid( x509Cert ) ) )
				{
					PGPByte		fingerprint[32];
					PGPSize		fingerLen;
					
					err = PGPGetKeyDBObjDataProperty(
							pgpKey, kPGPKeyProperty_Fingerprint, fingerprint,
							sizeof( fingerprint ), &fingerLen); CKERR;
					*mp++ = kPGPike_ID_Key_ID;			/* ID type */
					*mp++ = 0;							/* protocol ID */
					*mp++ = 0;*mp++ = 0;				/* port */
					pgpCopyMemory( fingerprint, mp, fingerLen );
					mp += fingerLen;
				}
				else if( PGPKeyDBObjRefIsValid( x509Cert ) )
				{
					PGPByte *dnBuffer = NULL;
					PGPSize	dnBufferLen;
					
					err = PGPGetKeyDBObjAllocatedDataProperty( x509Cert,
								kPGPSigProperty_X509DERDName,
								(void **)&dnBuffer, &dnBufferLen );
					if( IsntPGPError( err ) && IsntNull( dnBuffer ) )
					{
						*mp++ = kPGPike_ID_DER_ASN1_DN;		/* ID type */
						*mp++ = 0;							/* protocol ID */
						*mp++ = 0;*mp++ = 0;				/* port */
						pgpCopyMemory( dnBuffer, mp, dnBufferLen );
						mp += dnBufferLen;
					}
					if( IsntNull( dnBuffer ) )
						PGPFreeData( dnBuffer );
				}
				else
				{
					err = kPGPError_PublicKeyNotFound;
					goto done;
				}
				break;
			}
			case kPGPike_AM_PreSharedKey:
			default:
				if( IsntNull( exchange->partner->idData ) )
				{
					*mp++ = exchange->partner->idType;				/* ID type */
					*mp++ = 0;										/* protocol ID */
					*mp++ = 0;*mp++ = 0;							/* port */
					pgpCopyMemory( exchange->partner->idData, mp,
									exchange->partner->idDataSize );
					mp += exchange->partner->idDataSize;
				}
				else
				{
					*mp++ = kPGPike_ID_IPV4_Addr;
					*mp++ = 0;										/* protocol ID */
					*mp++ = 0;*mp++ = 0;							/* port */
					pgpCopyMemory( &exchange->partner->localIPAddress, mp, sizeof(PGPUInt32) );
					mp += sizeof(PGPUInt32);
				}
				break;
		}
		
		if( IsntNull( *pBody ) )
		{
			(void)PGPFreeData( *pBody );
			*pBody = NULL;
		}
	}
	*pSize = mp - *mpp;
	if( *pSize )
	{
		*pBody = PGPNewData( exchange->ike->memMgr,
					*pSize, kPGPMemoryMgrFlags_Clear );
		if( IsNull( *pBody ) )
		{
			err = kPGPError_OutOfMemory;
			goto done;
		}
		pgpCopyMemory( *mpp, *pBody, *pSize );
	}
done:
	*mpp = mp;
	return err;
}

	PGPError
pgpIKEAddCertPayload(
	PGPikeExchange *		exchange,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	PGPKeyDBObjRef			pgpKey;
	PGPKeyDBObjRef			x509Cert;
	PGPByte *				certBuffer = NULL;
	PGPSize					certBufferLen = 0;
	PGPContextRef			con = exchange->ike->pgpContext;
	
	pgpKey = exchange->partner->pgpAuthKey.authObj;
	x509Cert = exchange->partner->x509AuthKey.authObj;
	if( ( exchange->partner->remotePGPVersion && PGPKeyDBObjRefIsValid( pgpKey ) ) ||
		( ( exchange->exchangeT == kPGPike_EX_Aggressive ) &&
			PGPKeyDBObjRefIsValid( pgpKey ) && !PGPKeyDBObjRefIsValid( x509Cert ) ) )
	{
		*mp++ = kPGPike_CE_PGP;
		err = PGPExport( con,
				PGPOExportKeyDBObj( con, pgpKey ),
				PGPOAllocatedOutputBuffer( con, (void **) &certBuffer,
					MAX_PGPUInt32, &certBufferLen ),
				PGPOArmorOutput( con, FALSE ),
				PGPOLastOption( con ) );	CKERR;
	}
	else if( PGPKeyDBObjRefIsValid( x509Cert ) )
	{
		*mp++ = kPGPike_CE_X509_Sig;
		err = PGPExport( con,
				PGPOExportKeyDBObj( con, x509Cert ),
				PGPOExportFormat( con, kPGPExportFormat_X509Cert ),
				PGPOAllocatedOutputBuffer( con, (void **) &certBuffer,
					MAX_PGPUInt32, &certBufferLen ),
				PGPOLastOption( con ) );	CKERR;
	}
	else
	{
		err = kPGPError_PublicKeyNotFound;
		goto done;
	}
	if( ( certBufferLen > 0 ) && ( certBufferLen < MAX_PGPInt16 ) )	/* no DDT style keys!! */
	{
		pgpCopyMemory( certBuffer, mp, certBufferLen );
		mp += certBufferLen;
	}
	else
	{
		err = kPGPError_KeyUnusableForSignature;
		goto done;
	}
done:
	*mpp = mp;
	if( IsntNull( certBuffer ) )
		(void)PGPFreeData( certBuffer );
	return err;
}

	PGPError
pgpIKEAddCertRequestPayload(
	PGPikeExchange *		exchange,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	PGPKeyDBObjRef			pgpKey;
	PGPKeyDBObjRef			x509Cert;
	
	pgpKey = exchange->partner->pgpAuthKey.authObj;
	x509Cert = exchange->partner->x509AuthKey.authObj;
	if( ( exchange->partner->remotePGPVersion && PGPKeyDBObjRefIsValid( pgpKey ) ) ||
		( ( exchange->exchangeT == kPGPike_EX_Aggressive ) &&
			PGPKeyDBObjRefIsValid( pgpKey ) && !PGPKeyDBObjRefIsValid( x509Cert ) ) )
	{
		*mp++ = kPGPike_CE_PGP;
	}
	else if( PGPKeyDBObjRefIsValid( x509Cert ) )
	{
		*mp++ = kPGPike_CE_X509_Sig;
	}
	else
	{
		err = kPGPError_PublicKeyNotFound;
		goto done;
	}
done:
	*mpp = mp;
	return err;
}

	PGPError
pgpIKEAddSigPayload(
	PGPikeExchange *		exchange,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	PGPByte					authHash[kPGPike_MaxHashSize];
	PGPikeMTCert *			certInfo;
	PGPPrivateKeyContextRef	privKeyCon = kInvalidPGPPrivateKeyContextRef;
	PGPSize					sigSize,
							actSigSize;
	
	if( exchange->partner->remotePGPVersion &&
		PGPKeyDBObjRefIsValid( exchange->partner->pgpAuthKey.authObj ) )
		certInfo = &exchange->partner->pgpAuthKey;
	else
		certInfo = &exchange->partner->x509AuthKey;
	/* There's only one SIG payload in IKE, it signs HASH_I or HASH_R */
	err = pgpIKEGetAuthHash( exchange, exchange->initiator, authHash );CKERR;

	err = PGPNewPrivateKeyContext( PGPPeekKeyDBObjKey( certInfo->authObj ),
		kPGPPublicKeyMessageFormat_IKE,
		&privKeyCon,
		IsntNull( certInfo->pass ) ?
			( certInfo->isPassKey ?
				PGPOPasskeyBuffer( exchange->ike->pgpContext,
									certInfo->pass, certInfo->passLength ) :
				PGPOPassphrase( exchange->ike->pgpContext, certInfo->pass ) ) :
			PGPONullOption( exchange->ike->pgpContext ),
		PGPOLastOption( exchange->ike->pgpContext ) );	CKERR;
	err = PGPGetPrivateKeyOperationSizes( privKeyCon, NULL, NULL, &sigSize ); CKERR;
	err = PGPPrivateKeySignRaw( privKeyCon, authHash,
								exchange->partner->hashSize,
								mp, &actSigSize );
#if PGPIKE_VERBOSE
	if( IsPGPError( err ) )
		pgpIKEDebugLog( exchange->ike, TRUE, "PGPPrivateKeySignRaw PGPError: %ld\n", err );
#endif
	CKERR;
	pgpAssert( actSigSize == sigSize );
	mp += sigSize;
	
done:
	*mpp = mp;
	if( PGPPrivateKeyContextRefIsValid( privKeyCon ) )
		(void)PGPFreePrivateKeyContext( privKeyCon );
	return err;
}

	PGPError
pgpIKEAddKEPayload(
	PGPikeExchange *		exchange,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	PGPByte *				secretX = NULL;
	PGPUInt32				modBits,
							modBytes,
							expBytes = 0,
							expBits,
							ySize;
	
	/* Generate the secret random X value */
	modBits	 = PGPBigNumGetSignificantBits( exchange->dhP );
	modBytes = ( modBits + 7 ) / 8;
	(void)PGPDiscreteLogExponentBits( modBits, &expBytes );
	expBytes = ( expBytes * 3 / 2 + 7 ) / 8;
	expBits  = 8 * expBytes;
	secretX  = PGPNewSecureData( exchange->ike->memMgr, expBytes, 0 );
	if( IsNull( secretX ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}
	err = PGPContextGetRandomBytes( exchange->ike->pgpContext,
									secretX, expBytes ); CKERR;
	secretX[0] |= 0x80;
	err = PGPNewBigNum( exchange->ike->pgpContext, TRUE,
						&exchange->dhX ); CKERR;
	err = PGPBigNumInsertBigEndianBytes( exchange->dhX, secretX, 0,
											expBytes ); CKERR;
	if( exchange->initiator )
	{
		/* Generate initiator Y value */
		err = PGPNewBigNum( exchange->ike->pgpContext, TRUE,
							&exchange->dhYi );CKERR;
		err = PGPBigNumExpMod( exchange->dhG, exchange->dhX,
								exchange->dhP, exchange->dhYi ); CKERR;
		
		/* Write out just the Y value */
		ySize = ( ( PGPBigNumGetSignificantBits( exchange->dhP ) + 7 ) / 8 );
		err = PGPBigNumExtractBigEndianBytes( exchange->dhYi, mp, 0,
												ySize ); CKERR;
	}
	else
	{
		/* Generate responder Y value */
		err = PGPNewBigNum( exchange->ike->pgpContext, TRUE,
								&exchange->dhYr );CKERR;
		err = PGPBigNumExpMod( exchange->dhG, exchange->dhX,
								exchange->dhP, exchange->dhYr ); CKERR;
		
		/* Write out just the Y value */
		ySize = ( ( PGPBigNumGetSignificantBits( exchange->dhP ) + 7 ) / 8 );
		err = PGPBigNumExtractBigEndianBytes( exchange->dhYr, mp, 0,
												ySize ); CKERR;
	}
	mp += ySize;
done:
	*mpp = mp;
	if( IsntNull( secretX ) )
		(void)PGPFreeData( secretX );
	return err;
}

	PGPError
pgpIKEAddNoncePayload(
	PGPikeExchange *		exchange,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	
	err = PGPContextGetRandomBytes( exchange->ike->pgpContext,
								mp, kPGPike_NonceSize ); CKERR;
	if( exchange->initiator )
	{
		pgpCopyMemory( mp, exchange->initNonce, kPGPike_NonceSize );
		exchange->initNonceLen = kPGPike_NonceSize;
	}
	else
	{
		pgpCopyMemory( mp, exchange->respNonce, kPGPike_NonceSize );
		exchange->respNonceLen = kPGPike_NonceSize;
		if( exchange->state == kPGPike_S_AM_WaitSA )
			err = pgpIKEGoSecure( exchange ); CKERR;
	}
	mp += kPGPike_NonceSize;
done:
	*mpp = mp;
	return err;
}

	PGPError
pgpIKEAddSAPayload(
	PGPikeExchange *		exchange,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	PGPByte	*				hdr;
	PGPUInt16				plen;
	PGPikeProposal *		proposal;

	/* doi */
	PGPUInt32ToEndian( (PGPUInt32) exchange->doi,
			kPGPBigEndian, mp );
	mp += sizeof(PGPUInt32);
	
	/* situation */
	switch( exchange->doi )
	{
		case kPGPike_DOI_IPSEC:
			*mp++ = 0;
			*mp++ = 0;
			*mp++ = 0;
			*mp++ = kPGPike_IPSEC_SIT_IdentityOnly;
			break;
		case kPGPike_DOI_IKE:
			/* behavior undefined by IETF */
			pgpAssert( 0 );
			break;
		default:
			break;
	}
	/* Proposal */
	for( proposal = exchange->proposals; IsntNull( proposal );
			proposal = proposal->nextProposal )
	{
		hdr = mp;
		
		mp += kPGPike_ISAKMPPayloadHeaderSize;
		err = pgpIKEAddProposalPayload( exchange, proposal, &mp ); CKERR;
		plen = mp - hdr;
		if( IsntNull( proposal->nextProposal ) )
			*hdr++ = kPGPike_PY_Proposal;
		else
			*hdr++ = 0;
		*hdr++ = 0; /* reserved */
		PGPUInt16ToEndian( plen, kPGPBigEndian, hdr );
	}
done:
	*mpp = mp;
	return err;
}

	PGPError
pgpIKEAddProposalPayload(
	PGPikeExchange *		exchange,
	PGPikeProposal *		proposal,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	PGPByte *				hdr;
	PGPByte					spiSize;
	PGPUInt16				tix,
							plen;

	*mp++ = proposal->number;
	*mp++ = proposal->protocol;
	switch( proposal->protocol )
	{
		case kPGPike_PR_IKE:
			*mp++ = 0;	/* no SPI in Main Mode */
			*mp++ = proposal->numTransforms;
			break;
		case kPGPike_PR_IPCOMP:
		case kPGPike_PR_ESP:
		case kPGPike_PR_AH:
			if( proposal->protocol == kPGPike_PR_IPCOMP )
				spiSize = sizeof(PGPUInt16);
			else
				spiSize = sizeof(PGPipsecSPI);
			*mp++ = spiSize;
			*mp++ = proposal->numTransforms;
			/* insert SPI */
			if( proposal->protocol == kPGPike_PR_IPCOMP )
			{
				PGPUInt16	compCPI = (PGPUInt16)proposal->t[0].u.ipsec.ipcomp.compAlg;
				
				PGPUInt16ToEndian( compCPI, kPGPBigEndian, mp );
			}
			else if( exchange->initiator )
				pgpCopyMemory( &proposal->initSPI[0], mp, spiSize );
			else
				pgpCopyMemory( &proposal->respSPI[0], mp, spiSize );
			mp += spiSize;
			break;
		default:
			pgpAssert(0);
			break;
	}
	
	/* add appropriate transform payloads */
	for( tix = 0; tix < proposal->numTransforms; tix++ )
	{
		hdr = mp;
		mp += kPGPike_ISAKMPPayloadHeaderSize;
		*mp++ = tix + 1;
		err = pgpIKEAddTransformPayload( proposal, &proposal->t[tix], &mp ); CKERR;
		plen = mp - hdr;
		if( proposal->numTransforms - tix > 1 )
			*hdr++ = kPGPike_PY_Transform;
		else
			*hdr++ = 0;
		*hdr++ = 0;	/* reserved */
		PGPUInt16ToEndian( plen, kPGPBigEndian, hdr );
	}
	
	*mpp = mp;
done:
	return err;
}

	PGPError
pgpIKEAddTransformPayload(
	PGPikeProposal *		proposal,
	PGPikeGenericTransform *transform,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	PGPikeAttributeType		lt, ld;
	
	switch( proposal->protocol )
	{
		case kPGPike_PR_IKE:
		{
			PGPikeTransform *t = &transform->u.ike;
			
			*mp++ = kPGPike_Key_IKE_Transform;
			*mp++ = 0;	/* reserved */
			*mp++ = 0;	/* reserved */
			pgpIKEAddAttribute( kPGPike_AT_EncryptAlg, t->cipher, NULL, 0, &mp );
			pgpIKEAddAttribute( kPGPike_AT_HashAlg, t->hash, NULL, 0, &mp );
			pgpIKEAddAttribute( kPGPike_AT_AuthMethod, t->authMethod, NULL, 0, &mp );
			/*pgpIKEAddAttribute( kPGPike_AT_GroupType, kPGPike_MODPGroupType, NULL, 0, &mp );*/
			pgpIKEAddAttribute( kPGPike_AT_GroupDesc, t->groupID, NULL, 0, &mp );
			lt = kPGPike_AT_LifeType;	ld = kPGPike_AT_LifeDuration;
			break;
		}
		case kPGPike_PR_AH:
		{
			PGPipsecAHTransform *t = &transform->u.ipsec.ah;
			
			*mp++ = t->authAlg;
			*mp++ = 0;	/* reserved */
			*mp++ = 0;	/* reserved */
			pgpIKEAddAttribute( kPGPike_AT_IPSEC_AuthAttr, t->authAttr, NULL, 0, &mp );
			pgpIKEAddAttribute( kPGPike_AT_IPSEC_Encapsulation, t->mode, NULL, 0, &mp );
			if( proposal->t[0].u.ipsec.groupID != kPGPike_GR_None )
			{
				pgpIKEAddAttribute( kPGPike_AT_IPSEC_GroupDesc,
					proposal->t[0].u.ipsec.groupID, NULL, 0, &mp );
			}
			lt = kPGPike_AT_IPSEC_LifeType;	ld = kPGPike_AT_IPSEC_Duration;
			break;
		}
		case kPGPike_PR_ESP:
		{
			PGPipsecESPTransform *t = &transform->u.ipsec.esp;
			
			*mp++ = t->cipher;
			*mp++ = 0;	/* reserved */
			*mp++ = 0;	/* reserved */
			if( t->authAttr != kPGPike_AA_None )
				pgpIKEAddAttribute( kPGPike_AT_IPSEC_AuthAttr, t->authAttr, NULL, 0, &mp );
			pgpIKEAddAttribute( kPGPike_AT_IPSEC_Encapsulation, t->mode, NULL, 0, &mp );
			if( proposal->t[0].u.ipsec.groupID != kPGPike_GR_None )
			{
				pgpIKEAddAttribute( kPGPike_AT_IPSEC_GroupDesc,
					proposal->t[0].u.ipsec.groupID, NULL, 0, &mp );
			}
			lt = kPGPike_AT_IPSEC_LifeType;	ld = kPGPike_AT_IPSEC_Duration;
			break;
		}
		case kPGPike_PR_IPCOMP:
		{
			PGPipsecIPCOMPTransform *t = &transform->u.ipsec.ipcomp;
			
			*mp++ = t->compAlg;
			*mp++ = 0;	/* reserved */
			*mp++ = 0;	/* reserved */
			/* do not send encapsulation here */
			lt = kPGPike_AT_IPSEC_LifeType;	ld = kPGPike_AT_IPSEC_Duration;
			break;
		}
		default:
			break;
	}
	if( proposal->kbLifeTime )
	{
		pgpIKEAddAttribute( lt, kPGPike_LT_Kilobytes, NULL, 0, &mp );
		pgpIKEAddAttribute( ld, proposal->kbLifeTime, NULL, 0, &mp );
	}
	if( proposal->secLifeTime )
	{
		pgpIKEAddAttribute( lt, kPGPike_LT_Seconds, NULL, 0, &mp );
		pgpIKEAddAttribute( ld, proposal->secLifeTime, NULL, 0, &mp );
	}
	
	*mpp = mp;
	return err;
}

	PGPError
pgpIKEAddAttrPayload(
	PGPikeExchange *		exchange,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;
	PGPikeConfigType		ct;
	PGPUInt16				transID,
							configIndex;
	
	pgpAssert( exchange->numConfigs > 0 );
	for( configIndex = 0; configIndex < exchange->numConfigs; configIndex++ )
	{
		if( !configIndex )
		{
			ct		= exchange->configs[configIndex].configType;
			transID	= exchange->configs[configIndex].transactionID;
			*mp++ = ct;
			*mp++ = 0;
			PGPUInt16ToEndian( transID, kPGPBigEndian, mp );
			mp += sizeof(PGPUInt16);
		}
		if( ( ct != exchange->configs[configIndex].configType ) ||
			( transID != exchange->configs[configIndex].transactionID ) )
		{
			/* this would be a code error, report this */
			pgpAssert( 0 );
			break;
		}
		pgpIKEAddAttribute( exchange->configs[configIndex].attribute,
							exchange->configs[configIndex].value,
							exchange->configs[configIndex].valueP,
							exchange->configs[configIndex].valueSize, &mp );
	}
	*mpp = mp;
	return err;
}

	void
pgpIKEAddAttribute(
	PGPikeAttributeType		attribute,
	PGPUInt32				value,
	PGPByte *				valueP,
	PGPUInt16				valueSize,
	PGPByte **				mpp )
{
	PGPByte *				mp = *mpp;
	PGPBoolean				variable = FALSE;
	
	if( ( value > 0x0000FFFF ) || IsntNull( valueP ) )
		variable = TRUE;
	PGPUInt16ToEndian( (PGPUInt16)attribute, kPGPBigEndian, mp );
	if( !variable )
		*mp |= 0x80;
	mp += sizeof(PGPUInt16);
	if( variable )
	{
		if( IsNull( valueP ) )
		{
			PGPUInt16ToEndian( sizeof(PGPUInt32), kPGPBigEndian, mp );
			mp += sizeof(PGPUInt16);
			PGPUInt32ToEndian( value, kPGPBigEndian, mp );
			mp += sizeof(PGPUInt32);
		}
		else
		{
			PGPUInt16ToEndian( valueSize, kPGPBigEndian, mp );
			mp += sizeof(PGPUInt16);
			pgpCopyMemory( valueP, mp, valueSize );
			mp += valueSize;
		}
	}
	else
	{
		PGPUInt16ToEndian( (PGPUInt16)value, kPGPBigEndian, mp );
		mp += sizeof(PGPUInt16);
	}
	
	*mpp = mp;
}

    PGPBoolean
IsNATTraversalVendorIDStringPresent(
    PGPByte			*packet, 
    PGPSize			mLen)
{
	PGPSize			pLen;
	PGPikePayload	payload,
					nextPayload;

	if( mLen < kPGPike_ISAKMPHeaderSize + kPGPike_ISAKMPPayloadHeaderSize )
		return FALSE;
	packet += kPGPike_CookieSize * 2;
	/* Next Payload */
	payload = (PGPikePayload)*packet;
	/* Skip NextPayload(1), Version(1), ExchangeType(1), Flags(1), and MessageID(4) */
	packet += 8;
	/* Length */
	pLen = PGPEndianToUInt32( kPGPBigEndian, packet );
	if( pLen > mLen )
		return FALSE;
	/* Skip Length */
	packet += 4;
	mLen -= kPGPike_ISAKMPHeaderSize;
	/* Check for Payloads */
	while( payload != kPGPike_PY_None )
	{
		if( mLen < kPGPike_ISAKMPPayloadHeaderSize )
			return FALSE;
		nextPayload = (PGPikePayload)*packet;
		/* Get Length at offset 2 */
		pLen = PGPEndianToUInt16( kPGPBigEndian, packet + 2 );
		if( mLen < pLen)
			return FALSE;

		if( payload == kPGPike_PY_VendorID )
		{
			PGPByte		*mp;
			PGPSize		bodyLen, vidSize;

			/* Skip isakmp Header */
			mp = packet + kPGPike_ISAKMPPayloadHeaderSize;
			bodyLen = pLen - kPGPike_ISAKMPPayloadHeaderSize;

			/* Check if this is the vendor id we need */
			vidSize = sizeof( kPGPike_NATTraversalVendorID );

			if( ( bodyLen == vidSize ) && pgpMemoryEqual(kPGPike_NATTraversalVendorID, mp, vidSize) )
				return TRUE;
		}
		payload = nextPayload;
		packet += pLen;
		mLen -= pLen;
	}
	return FALSE;
}

	PGPError
pgpIKEGetNATDiscoveryHash(
	PGPikeExchange *		exchange,
	PGPByte *				mp )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner = exchange->partner;
	PGPHashContextRef		hash = kInvalidPGPHashContextRef;
	PGPUInt16				nboPort;

	err = PGPNewHashContext( exchange->ike->pgpContext,
			kPGPHashAlgorithm_MD5, &hash );						CKERR;
	err = PGPContinueHash( hash, exchange->initiator ?
			&partner->localIPAddress : &partner->ipAddress,
			sizeof( PGPUInt32) );								CKERR;
	PGPUInt16ToEndian( exchange->initiator ?
			kPGPike_CommonPort : partner->destPort, kPGPBigEndian, (PGPByte *)&nboPort );
	err = PGPContinueHash( hash, (void *)&nboPort, sizeof( PGPUInt16) );CKERR;

	err = PGPContinueHash( hash, exchange->initiator ?
			&partner->ipAddress : &partner->localIPAddress,
			sizeof( PGPUInt32) );								CKERR;
	PGPUInt16ToEndian( exchange->initiator ?
			partner->destPort : kPGPike_CommonPort, kPGPBigEndian, (PGPByte *)&nboPort );
	err = PGPContinueHash( hash, (void *)&nboPort, sizeof( PGPUInt16) );CKERR;

	err = PGPFinalizeHash( hash, mp );							CKERR;
done:
	if( PGPHashContextRefIsValid( hash ) )
		(void) PGPFreeHashContext( hash );
	return err;
}

	PGPError
pgpIKEAddNATDiscoveryPayload(
	PGPikeExchange *		exchange,
	PGPByte **				mpp )
{
	PGPError				err = kPGPError_NoErr;
	PGPByte *				mp = *mpp;

	err = pgpIKEGetNATDiscoveryHash( exchange, mp );
	if( IsntPGPError( err ) )
		mp += kPGPike_MD5HashSize;
	*mpp = mp;
	return err;
}

	PGPError
pgpIKELoadGroup(
	PGPikeExchange *		exchange )
{
	PGPError				err = kPGPError_NoErr;
	IKEGroup *				group;
	PGPikeGroupID			groupID;
	PGPUInt16				gIndex;
	
	pgpAssert( IsNull( exchange->dhG ) );
	pgpAssert( IsNull( exchange->dhP ) );
	
	if( ( exchange->exchangeT == kPGPike_EX_Identity ) ||
		( exchange->exchangeT == kPGPike_EX_Aggressive ) )
		groupID = exchange->proposals->t[0].u.ike.groupID;
	else
		groupID = exchange->proposals->t[0].u.ipsec.groupID;

	for( gIndex = 0; gIndex < kPGPikeNumIKEGroups; gIndex++ )
		if( kPGPike_Groups[gIndex].ikeID == groupID )
			break;
	if( gIndex >= kPGPikeNumIKEGroups )
	{
		err = kPGPError_ItemNotFound;
		goto done;
	}
	group = &kPGPike_Groups[gIndex];
	
	/* Load up the prime modulus */
	err = PGPNewBigNum( exchange->ike->pgpContext, TRUE, &exchange->dhP );CKERR;
	err =  PGPBigNumInsertBigEndianBytes(	exchange->dhP, group->prime,
											0, group->length ); CKERR;
	/* Load up the generator (always 2) */
	err = PGPNewBigNum( exchange->ike->pgpContext, TRUE, &exchange->dhG );CKERR;
	err =  PGPBigNumInsertBigEndianBytes(	exchange->dhG, dhGenerator,
											0, sizeof(dhGenerator) ); CKERR;
done:
	return err;
}

	PGPError
pgpIKEModP(
	PGPikeExchange *		exchange )
{
	PGPError				err = kPGPError_NoErr;
	PGPBigNumRef			dhSecret = kPGPInvalidBigNumRef;
	PGPUInt32				modBytes;

	err = PGPNewBigNum( exchange->ike->pgpContext, TRUE, &dhSecret ); CKERR;
	err = PGPBigNumExpMod( exchange->initiator ?
								exchange->dhYr : exchange->dhYi,
							exchange->dhX, exchange->dhP,
							dhSecret ); CKERR;
	
	/* Load gXY with the shared secret */
	modBytes = ( PGPBigNumGetSignificantBits( exchange->dhP ) + 7 ) / 8;
	exchange->gXYLen = modBytes;
	exchange->gXY = PGPNewSecureData( exchange->ike->memMgr,
											exchange->gXYLen, 0 );
	if( IsNull( exchange->gXY ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}
	err = PGPBigNumExtractBigEndianBytes( dhSecret, exchange->gXY, 0,
											modBytes );	CKERR;
done:
	if( dhSecret != kPGPInvalidBigNumRef )
		(void)PGPFreeBigNum( dhSecret );
	return err;
}

	PGPError
pgpIKEGoSecure(
	PGPikeExchange *		exchange )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner = exchange->partner;
	PGPUInt16				ySize;
	PGPByte *				ivSeed = NULL;
	PGPHMACContextRef		hmac = kInvalidPGPHMACContextRef;
	PGPHMACContextRef		hmacE = kInvalidPGPHMACContextRef;
	PGPHashContextRef		hash = kInvalidPGPHashContextRef;
	PGPByte					hashChar;
	PGPByte					lastHash[kPGPike_MaxHashSize];
	PGPByte					nonceCat[kPGPike_MaxNonceSize * 2];
	PGPSize					nonceCatLen;
			
	PGPSymmetricCipherContextRef symmRef = kInvalidPGPSymmetricCipherContextRef;
	
	if( IsNull( exchange->dhP ) ||
		IsNull( exchange->dhG ) ||
		IsNull( exchange->dhX ) ||
		IsNull( exchange->dhYi ) ||
		IsNull( exchange->dhYr ) ||
		IsntNull( exchange->gXY ) ||
		( exchange->initNonceLen == 0 ) ||
		( exchange->respNonceLen == 0 ) )
	{
		pgpAssert( 0 );		/* these would be a programming error */
		err = kPGPError_ImproperInitialization;
		goto done;
	}
	err = pgpIKEModP( exchange );	CKERR;
#if PGPIKE_VERBOSE
	pgpIKEDebugLog( exchange->ike, TRUE, "\tpgpIKEGoSecure:\n" );
	pgpIKEDebugData( exchange->ike, "gXY", exchange->gXY, exchange->gXYLen );
#endif
	/* everything is in place, shared secret is set, generate SKEYID */
	switch( exchange->proposals->t[0].u.ike.hash )
	{
		case kPGPike_HA_MD5:
			partner->sdkHashAlg		=	kPGPHashAlgorithm_MD5;
			partner->hashSize		=	kPGPike_MD5HashSize;
			break;
		case kPGPike_HA_SHA1:
			partner->sdkHashAlg		=	kPGPHashAlgorithm_SHA;
			partner->hashSize		=	kPGPike_SHAHashSize;
			break;
		default:
		case kPGPike_HA_Tiger:
			pgpAssert( 0 );
			break;
	}
	switch( exchange->proposals->t[0].u.ike.cipher )
	{
		case kPGPike_SC_IDEA_CBC:
			partner->sdkCipherAlg	=	kPGPCipherAlgorithm_IDEA;
			partner->cipherSize		=	kPGPike_IDEAKeySize;
			break;
		case kPGPike_SC_DES_CBC:
			partner->singleDESMode	=	TRUE;
		case kPGPike_SC_3DES_CBC:
			partner->sdkCipherAlg	=	kPGPCipherAlgorithm_3DES;
			partner->cipherSize		=	kPGPike_3DESKeySize;
			break;
		case kPGPike_SC_CAST_CBC:
			partner->sdkCipherAlg	=	kPGPCipherAlgorithm_CAST5;
			partner->cipherSize		=	kPGPike_CASTKeySize;
			break;
		default:
			break;
	}
	partner->cipherBlockSize		=	kPGPike_MaxCipherBlockSize;
#if PGPIKE_VERBOSE
	pgpIKEDebugData( exchange->ike, "Ni", exchange->initNonce,
							exchange->initNonceLen );
	pgpIKEDebugData( exchange->ike, "Nr", exchange->respNonce,
							exchange->respNonceLen );
#endif
	pgpCopyMemory( exchange->initNonce, nonceCat,
					exchange->initNonceLen );
	pgpCopyMemory( exchange->respNonce,
					nonceCat + exchange->initNonceLen,
					exchange->respNonceLen );
	nonceCatLen = exchange->initNonceLen + exchange->respNonceLen;
	switch( exchange->proposals->t[0].u.ike.authMethod )
	{
		case kPGPike_AM_DSS_Sig:
		case kPGPike_AM_RSA_Sig:
		{
			err = PGPNewHMACContext( exchange->ike->pgpContext,
							partner->sdkHashAlg,
							nonceCat, nonceCatLen, &hmac );	CKERR;
			err = PGPContinueHMAC( hmac, exchange->gXY,
									exchange->gXYLen );	CKERR;
			err = PGPFinalizeHMAC( hmac, partner->skeyid ); CKERR;
			(void)PGPFreeHMACContext( hmac );
			hmac = kInvalidPGPHMACContextRef;
			break;
		}
		case kPGPike_AM_PreSharedKey:
		{
			pgpAssert( IsntNull( partner->sharedKey ) &&
						( partner->sharedKeySize > 0 ) );
			err = PGPNewHMACContext( exchange->ike->pgpContext,
							partner->sdkHashAlg,
							partner->sharedKey, partner->sharedKeySize,
							&hmac );	CKERR;
			err = PGPContinueHMAC( hmac, nonceCat, nonceCatLen );	CKERR;
			err = PGPFinalizeHMAC( hmac, partner->skeyid ); CKERR;
			(void)PGPFreeHMACContext( hmac );
			hmac = kInvalidPGPHMACContextRef;
			break;
		}
		case kPGPike_AM_RSA_Encrypt:
		case kPGPike_AM_RSA_Encrypt_R:
			pgpAssert( 0 );
			break;
		default:
			break;
	}
#if PGPIKE_VERBOSE
	pgpIKEDebugData( exchange->ike, "SKEYID", partner->skeyid,
							partner->hashSize );
#endif
	/* SKEYID_d */
	err = PGPNewHMACContext( exchange->ike->pgpContext,
					partner->sdkHashAlg,
					partner->skeyid,
					partner->hashSize, &hmac );CKERR;
	err = PGPContinueHMAC( hmac, exchange->gXY, exchange->gXYLen );CKERR;
	err = PGPContinueHMAC( hmac, partner->initCookie,
							kPGPike_CookieSize );CKERR;
	err = PGPContinueHMAC( hmac, partner->respCookie,
							kPGPike_CookieSize );CKERR;
	hashChar = 0x00;
	err = PGPContinueHMAC( hmac, &hashChar, 1 );CKERR;
	err = PGPFinalizeHMAC( hmac, partner->skeyid_d ); CKERR;
	/* SKEYID_a */
	err = PGPResetHMAC( hmac );	CKERR;
	err = PGPContinueHMAC( hmac, partner->skeyid_d,
									partner->hashSize );CKERR;
	err = PGPContinueHMAC( hmac, exchange->gXY, exchange->gXYLen );CKERR;
	err = PGPContinueHMAC( hmac, partner->initCookie,
							kPGPike_CookieSize );CKERR;
	err = PGPContinueHMAC( hmac, partner->respCookie,
							kPGPike_CookieSize );CKERR;
	hashChar = 0x01;
	err = PGPContinueHMAC( hmac, &hashChar, 1 );CKERR;
	err = PGPFinalizeHMAC( hmac, partner->skeyid_a ); CKERR;
	/* SKEYID_e */
	err = PGPResetHMAC( hmac );	CKERR;
	err = PGPContinueHMAC( hmac, partner->skeyid_a,
									partner->hashSize );CKERR;
	err = PGPContinueHMAC( hmac, exchange->gXY, exchange->gXYLen );CKERR;
	err = PGPContinueHMAC( hmac, partner->initCookie,
							kPGPike_CookieSize );CKERR;
	err = PGPContinueHMAC( hmac, partner->respCookie,
							kPGPike_CookieSize );CKERR;
	hashChar = 0x02;
	err = PGPContinueHMAC( hmac, &hashChar, 1 );CKERR;
	err = PGPFinalizeHMAC( hmac, partner->skeyid_e ); CKERR;
#if PGPIKE_VERBOSE
	pgpIKEDebugData( exchange->ike, "SKEYID_d", partner->skeyid_d,
							partner->hashSize );
	pgpIKEDebugData( exchange->ike, "SKEYID_a", partner->skeyid_a,
							partner->hashSize );
	pgpIKEDebugData( exchange->ike, "SKEYID_e", partner->skeyid_e,
							partner->hashSize );
#endif
	
	/* This whole protocol, mostly just to get this one key.... */
	if( !partner->singleDESMode && ( partner->cipherSize > partner->hashSize ) )
	{
		PGPUInt16			keySoFar = 0;
		PGPSize				lastHashLen = 1;
		
		err = PGPNewHMACContext( exchange->ike->pgpContext,
									partner->sdkHashAlg,
									partner->skeyid_e,
									partner->hashSize, &hmacE );CKERR;
		while( keySoFar < partner->cipherSize )
		{
			if( !keySoFar )
				lastHash[0] = 0x00;
			err = PGPContinueHMAC( hmacE, &lastHash, lastHashLen );CKERR;
			err = PGPFinalizeHMAC( hmacE, lastHash ); CKERR;
			lastHashLen = partner->hashSize;
#if PGPIKE_VERBOSE
			pgpIKEDebugData( exchange->ike, "KX", lastHash, lastHashLen );
#endif
			if( keySoFar + lastHashLen > partner->cipherSize )
				lastHashLen = partner->cipherSize - keySoFar;
			pgpCopyMemory( lastHash, partner->cipherKey + keySoFar,
							lastHashLen );
			keySoFar += lastHashLen;
			err = PGPResetHMAC( hmacE );	CKERR;
		}
		err = PGPFreeHMACContext( hmacE );	CKERR;
		hmacE = kInvalidPGPHMACContextRef;
	}
	else
		pgpCopyMemory( partner->skeyid_e, partner->cipherKey,
						partner->cipherSize );
	if( partner->singleDESMode )
	{
		pgpClearMemory( partner->cipherKey + kPGPike_DESKeySize,
						kPGPike_3DESKeySize - kPGPike_DESKeySize );
	}
#if PGPIKE_VERBOSE
	pgpIKEDebugData( exchange->ike, "cipherKey", partner->cipherKey,
							partner->cipherSize );
#endif
	if( PGPCBCContextRefIsValid( partner->cbc ) )
	{
		PGPFreeCBCContext( partner->cbc );
		partner->cbc = kInvalidPGPCBCContextRef;
	}
	/* Key is all set, setup the CBC context */
	err = PGPNewSymmetricCipherContext( exchange->ike->pgpContext,
							partner->sdkCipherAlg,
							&symmRef );	CKERR;
	err = PGPNewCBCContext( symmRef, &partner->cbc ); CKERR;
	symmRef = kInvalidPGPSymmetricCipherContextRef;
	/* create the initial IV for the CBC context */
	ySize = ( ( PGPBigNumGetSignificantBits( exchange->dhP ) + 7 ) / 8 );
	ivSeed = PGPNewData( exchange->ike->memMgr, ySize * 2,
							kPGPMemoryMgrFlags_Clear );
	if( IsNull( ivSeed ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}
	err = PGPBigNumExtractBigEndianBytes( exchange->dhYi, ivSeed, 0,
											ySize ); CKERR;
	err = PGPBigNumExtractBigEndianBytes( exchange->dhYr, ivSeed + ySize, 0,
											ySize ); CKERR;
	err = PGPNewHashContext( exchange->ike->pgpContext, partner->sdkHashAlg,
								&hash );	CKERR;
	err = PGPContinueHash( hash, ivSeed, ySize * 2 );	CKERR;
	err = PGPFinalizeHash( hash, lastHash );	CKERR;
	pgpCopyMemory( lastHash, exchange->lastCBC, partner->cipherBlockSize );
	pgpCopyMemory( lastHash, partner->lastP1CBC, partner->cipherBlockSize );
#if PGPIKE_VERBOSE
	pgpIKEDebugData( exchange->ike, "IV", partner->lastP1CBC,
							partner->cipherBlockSize );
#endif
	
done:
	if( IsntNull( ivSeed ) )
		(void)PGPFreeData( ivSeed );
	if( PGPHashContextRefIsValid( hash ) )
		(void)PGPFreeHashContext( hash );
	if( PGPHMACContextRefIsValid( hmac ) )
		(void)PGPFreeHMACContext( hmac );
	if( PGPHMACContextRefIsValid( hmacE ) )
		(void)PGPFreeHMACContext( hmacE );
	if( PGPSymmetricCipherContextRefIsValid( symmRef ) )
		(void)PGPFreeSymmetricCipherContext( symmRef );
	return err;
}

	PGPError
pgpIKESAEstablished(
	PGPikePartner *			partner,
	PGPipsecSA *			sa )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeMTSAEstablished	sae;
	PGPipsecSA				*walkSA,
							*lowBirthSA;
	PGPContextRef			con = partner->ike->pgpContext;
	PGPBoolean				found = FALSE;
	PGPUInt32				saCount;
#if PGPIKE_DEBUG
	PGPUInt16				tInx;
#endif
	
	err = pgpIKERemovePending( partner->ike, sa->ipAddress, sa->ipAddrStart,
			sa->ipMaskEnd, sa->destIsRange );	CKERR;
	/* Send the identity message if necessary */
	found = FALSE;
	for( walkSA = partner->ike->sa; IsntNull( walkSA ); walkSA = walkSA->nextSA )
		if( ( walkSA != sa ) &&
			( walkSA->ipAddress == partner->ipAddress ) && walkSA->activeOut )
			found = TRUE;
	if( !found && partner->assignedIP )
	{
		PGPikeMTIdentity	mtIdentity;
		
		pgpClearMemory( &mtIdentity, sizeof(PGPikeMTIdentity) );
		mtIdentity.active		= TRUE;
		mtIdentity.ipAddress	= partner->ipAddress;
		mtIdentity.assignedIP	= partner->assignedIP;
		mtIdentity.assignedDNS	= partner->assignedDNS;
		mtIdentity.assignedWINS	= partner->assignedWINS;
		err = (partner->ike->msgProc)(
				(PGPikeContextRef)partner->ike, partner->ike->userData,
				kPGPike_MT_Identity, &mtIdentity );	CKERR;
	}
	/* Send the SA Established message */
	sae.sa = sa;
	if( PGPKeyDBObjRefIsValid( partner->remoteObj ) )
	{
		sae.remoteValid = partner->remoteValid;
		err = PGPExport( con,
					PGPOExportKeyDBObj( con, partner->remoteObj ),
					PGPOAllocatedOutputBuffer( con,
						(void **)&sae.remoteAuthKey, MAX_PGPSize, &sae.remoteAuthKeySize ),
					PGPOArmorOutput( con, FALSE ),
					PGPOLastOption( con ) );	CKERR;
	}
	else
	{
		sae.remoteValid = TRUE;
		sae.remoteAuthKey = NULL;
		sae.remoteAuthKeySize = 0;
	}
	err = (partner->ike->msgProc)( (PGPikeContextRef)partner->ike,
				partner->ike->userData, kPGPike_MT_SAEstablished, &sae );
	if( IsntNull( sae.remoteAuthKey ) )
		(void) PGPFreeData( sae.remoteAuthKey );
#if PGPIKE_DEBUG
	{
		char	ip1Str[20],
				ip2Str[20];
		
		pgpIKEGetIPString( sa->ipAddress, ip1Str );
		pgpIKEDebugLog( partner->ike, TRUE, "NEWP2 - IP: %s   Life: S: %u    K: %u\n",
				ip1Str, sa->secLifeTime, sa->kbLifeTime );
		pgpIKEGetIPString( sa->ipAddrStart, ip1Str );
		pgpIKEGetIPString( sa->ipMaskEnd, ip2Str );
		pgpIKEDebugLog( partner->ike, FALSE, "\t# Protocols: %u\tDest - IP: %s -> %s (%s)\n",
				sa->numTransforms, ip1Str, ip2Str,
				sa->destIsRange ? "Range" : "Mask" );
		for( tInx = 0; tInx < sa->numTransforms; tInx++ )
		{
			pgpIKEDebugData( partner->ike, "SPI Inbound",
				(PGPByte *)&sa->transform[tInx].inSPI, sizeof(PGPipsecSPI) );
			pgpIKEDebugData( partner->ike, "SPI Outbound",
				(PGPByte *)&sa->transform[tInx].outSPI, sizeof(PGPipsecSPI) );
			pgpIKEDebugLog( partner->ike, FALSE, "\t(%u) Prot: %s\n",
					tInx, pgpIKEProtocolString( sa->transform[tInx].protocol ) );
		}
	}
#endif
	err = pgpIKELocalAlert( partner->ike, sa->ipAddress,
			kPGPike_AL_None, kPGPike_IA_NewPhase2SA, FALSE );
	
	lowBirthSA = NULL;
	saCount = 1;
	for( walkSA = partner->ike->sa; IsntNull( walkSA ); walkSA = walkSA->nextSA )
	{
		if( ( walkSA != sa ) &&
			( walkSA->ipAddress == sa->ipAddress ) && walkSA->activeOut &&
			( walkSA->destIsRange == sa->destIsRange ) &&
			( walkSA->ipAddrStart == sa->ipAddrStart ) &&
			( walkSA->ipMaskEnd == sa->ipMaskEnd ) &&
			( walkSA->ipProtocol == sa->ipProtocol ) &&
			( walkSA->ipPort == sa->ipPort ) )
		{
			saCount++;
			if( !lowBirthSA || ( lowBirthSA->birthTime > walkSA->birthTime ) )
				lowBirthSA = walkSA;
		}
	}
	if( saCount > kPGPike_MaxIdenticalSAs )
	{
		err = pgpIKEKillSA( &partner, lowBirthSA );	CKERR;
	}
done:
	return err;
}

	PGPError
pgpIKEGetP2Keymat(
	PGPikeExchange *		exchange,
	PGPipsecDOIParams *		ipsec )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner = exchange->partner;
	PGPUInt16				keymatNeeded = 0,
							keymatIters,
							keymatIndex;
	PGPHMACContextRef		hmac = kInvalidPGPHMACContextRef;
	PGPByte					inKeymat[256];
	PGPByte					outKeymat[256];
	PGPByte *				iKeymatP;
	PGPByte *				oKeymatP;
	PGPByte					protocol8;
	PGPSize					keySize;
	
	/* This function could be somewhat cleaner */
	if( IsNull( exchange->gXY ) &&
		( exchange->proposals->t[0].u.ipsec.groupID != kPGPike_GR_None ) )
	{
		if( IsntNull( exchange->dhP ) &&
			IsntNull( exchange->dhG ) &&
			IsntNull( exchange->dhX ) &&
			IsntNull( exchange->dhYi ) &&
			IsntNull( exchange->dhYr ) )
		{
			err = pgpIKEModP( exchange );	CKERR;
#if PGPIKE_DEBUG
			pgpIKEDebugLog( exchange->ike, TRUE, "\tQM Key Material using PFS.\n" );
#endif
		}
		else
		{
			pgpAssert( 0 );		/* these would be a programming error */
			err = kPGPError_ImproperInitialization;
			goto done;
		}
	}
	switch( ipsec->protocol )
	{
		case kPGPike_PR_AH:
			switch( ipsec->u.ah.t.authAttr )
			{
				case kPGPike_AA_HMAC_MD5:
					keymatNeeded += kPGPike_MD5HashSize;
					break;
				case kPGPike_AA_HMAC_SHA:
					keymatNeeded += kPGPike_SHAHashSize;
					break;
				default:
					pgpAssert( 0 );	/* unsupported AH attr */
					break;
			}
			break;
		case kPGPike_PR_ESP:
			switch( ipsec->u.esp.t.cipher )
			{
				case kPGPike_ET_3DES:
					keymatNeeded += kPGPike_3DESKeySize;
					break;
				case kPGPike_ET_CAST:
					keymatNeeded += kPGPike_CASTKeySize;
					break;
				case kPGPike_ET_DES:
				case kPGPike_ET_DES_IV64:
					keymatNeeded += kPGPike_DESKeySize;
					break;
				case kPGPike_ET_NULL:
					break;
				default:
					pgpAssert( 0 );	/* unsupported cipher */
					break;
			}
			switch( ipsec->u.ah.t.authAttr )
			{
				case kPGPike_AA_HMAC_MD5:
					keymatNeeded += kPGPike_MD5HashSize;
					break;
				case kPGPike_AA_HMAC_SHA:
					keymatNeeded += kPGPike_SHAHashSize;
					break;
				case kPGPike_AA_None:
					break;
				default:
					pgpAssert( 0 );	/* unsupported AH attr */
					break;
			}
			break;
		case kPGPike_PR_IPCOMP:
			goto done;	/* no keymat required */
		default:
			break;
	}
	pgpAssert( keymatNeeded < 256 );
	keymatIters = keymatNeeded / partner->hashSize;
	if( keymatNeeded % partner->hashSize )
		keymatIters++;
	protocol8 = (PGPByte)ipsec->protocol;
	err = PGPNewHMACContext( exchange->ike->pgpContext,
							partner->sdkHashAlg,
							partner->skeyid_d, partner->hashSize, &hmac );	CKERR;
	iKeymatP = inKeymat;
	for( keymatIndex = 0; keymatIndex < keymatIters; keymatIndex++ )
	{
		if( keymatIndex > 0 )
		{
			err = PGPContinueHMAC( hmac, iKeymatP, partner->hashSize );	CKERR;
			iKeymatP += partner->hashSize;
		}
		if( IsntNull( exchange->gXY ) )
		{
			err = PGPContinueHMAC( hmac, exchange->gXY,
								exchange->gXYLen );	CKERR;
		}
		err = PGPContinueHMAC( hmac, &protocol8, 1 );
		err = PGPContinueHMAC( hmac, &ipsec->inSPI, sizeof(PGPipsecSPI) );	CKERR;
		err = PGPContinueHMAC( hmac, &exchange->initNonce,
									exchange->initNonceLen );	CKERR;
		err = PGPContinueHMAC( hmac, &exchange->respNonce,
									exchange->respNonceLen );	CKERR;
		err = PGPFinalizeHMAC( hmac, iKeymatP );	CKERR;
		err = PGPResetHMAC( hmac );	CKERR;
	}
	err = PGPResetHMAC( hmac );	CKERR;
	oKeymatP = outKeymat;
	for( keymatIndex = 0; keymatIndex < keymatIters; keymatIndex++ )
	{
		if( keymatIndex > 0 )
		{
			err = PGPContinueHMAC( hmac, oKeymatP, partner->hashSize );	CKERR;
			oKeymatP += partner->hashSize;
		}
		if( IsntNull( exchange->gXY ) )
		{
			err = PGPContinueHMAC( hmac, exchange->gXY,
								exchange->gXYLen );	CKERR;
		}
		err = PGPContinueHMAC( hmac, &protocol8, 1 );
		err = PGPContinueHMAC( hmac, &ipsec->outSPI, sizeof(PGPipsecSPI) );	CKERR;
		err = PGPContinueHMAC( hmac, &exchange->initNonce,
									exchange->initNonceLen );	CKERR;
		err = PGPContinueHMAC( hmac, &exchange->respNonce,
									exchange->respNonceLen );	CKERR;
		err = PGPFinalizeHMAC( hmac, oKeymatP );	CKERR;
		err = PGPResetHMAC( hmac );	CKERR;
	}
	iKeymatP = inKeymat;
	oKeymatP = outKeymat;
	switch( ipsec->protocol )
	{
		case kPGPike_PR_AH:
			switch( ipsec->u.ah.t.authAttr )
			{
				case kPGPike_AA_HMAC_MD5:
					keySize = kPGPike_MD5HashSize;
					break;
				case kPGPike_AA_HMAC_SHA:
					keySize = kPGPike_SHAHashSize;
					break;
				default:
					pgpAssert( 0 );
			}
			pgpCopyMemory( iKeymatP, ipsec->u.ah.inAuthKey, keySize );
			pgpCopyMemory( oKeymatP, ipsec->u.ah.outAuthKey, keySize );
			iKeymatP += keySize;
			oKeymatP += keySize;
			break;
		case kPGPike_PR_ESP:
			err = PGPContextGetRandomBytes( exchange->ike->pgpContext,
					&ipsec->u.esp.explicitIV,
					kPGPike_MaxExplicitIVSize ); CKERR;
			switch( ipsec->u.esp.t.cipher )
			{
				case kPGPike_ET_3DES:
					keySize = kPGPike_3DESKeySize;
					break;
				case kPGPike_ET_CAST:
					keySize = kPGPike_CASTKeySize;
					break;
				case kPGPike_ET_DES_IV64:
				case kPGPike_ET_DES:
					keySize = kPGPike_DESKeySize;
					break;
				case kPGPike_ET_NULL:
					keySize = 0;
					break;
				default:
					pgpAssert( 0 );
			}
			pgpCopyMemory( iKeymatP, ipsec->u.esp.inESPKey, keySize );
			pgpCopyMemory( oKeymatP, ipsec->u.esp.outESPKey, keySize );
			iKeymatP += keySize;
			oKeymatP += keySize;
			switch( ipsec->u.ah.t.authAttr )
			{
				case kPGPike_AA_HMAC_MD5:
					keySize = kPGPike_MD5HashSize;
					break;
				case kPGPike_AA_HMAC_SHA:
					keySize = kPGPike_SHAHashSize;
					break;
				case kPGPike_AA_None:
					keySize = 0;
					break;
				default:
					pgpAssert( 0 );
			}
			pgpCopyMemory( iKeymatP, ipsec->u.esp.inAuthKey, keySize );
			pgpCopyMemory( oKeymatP, ipsec->u.esp.outAuthKey, keySize );
			iKeymatP += keySize;
			oKeymatP += keySize;
			break;
		default:
			break;
	}
done:
	if( PGPHMACContextRefIsValid( hmac ) )
		(void)PGPFreeHMACContext( hmac );
	return err;
}

	PGPError
pgpIKECompletePhase2(
	PGPikeExchange *		exchange,
	PGPipsecSA **			saP )
{
	PGPError				err = kPGPError_NoErr;
	PGPikePartner *			partner = exchange->partner;
	PGPipsecSA *			sa = NULL;
	PGPikeProposal *		proposal;
	PGPUInt16				propNum = 0;
	
	*saP = NULL;
	sa = PGPNewData( exchange->ike->memMgr, sizeof(PGPipsecSA),
						kPGPMemoryMgrFlags_Clear );
	if( IsNull( sa ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}
	/* DISABLED NAT TRAV FOR 71
	sa->bNATTraversal	= partner->bNATTraversal;
	if( partner->bNATTraversal && partner->preNATIPAddress )
		sa->ipAddress = partner->preNATIPAddress;
	else*/
		sa->ipAddress = partner->ipAddress;

    /*DISABLED NAT TRAV FOR 71  sa->natIPAddress = partner->ipAddress;*/
	PGPUInt16ToEndian( partner->destPort, kPGPBigEndian, (PGPByte *)&sa->nboNATPort);
	sa->initiator		= exchange->initiator;
	sa->activeIn		= TRUE;
	sa->activeOut		= TRUE;
	sa->kbLifeTime		= exchange->proposals->kbLifeTime;
	sa->secLifeTime		= exchange->proposals->secLifeTime;
	sa->birthTime		= PGPGetTime();
	pgpAssert( IsntNull( exchange->destination ) );
	/* This should never be NULL */
	if( IsntNull( exchange->destination ) )
	{
		sa->destIsRange		= exchange->destination->destIsRange;
		sa->ipAddrStart		= exchange->destination->ipAddrStart;
		sa->ipMaskEnd		= exchange->destination->ipMaskEnd;
		sa->ipPort			= exchange->destination->ipPort;
		sa->ipProtocol		= exchange->destination->ipProtocol;
	}
	sa->assignedIP		= partner->assignedIP;
	sa->assignedDNS		= partner->assignedDNS;
	sa->assignedWINS	= partner->assignedWINS;
	for( proposal = exchange->proposals; IsntNull( proposal );
			proposal = proposal->nextProposal )
	{
		sa->transform[propNum].protocol	= proposal->protocol;
		if( exchange->initiator )
		{
			pgpCopyMemory( proposal->respSPI, sa->transform[propNum].outSPI,
							sizeof(PGPipsecSPI) );
			pgpCopyMemory( proposal->initSPI, sa->transform[propNum].inSPI,
							sizeof(PGPipsecSPI) );
		}
		else
		{
			pgpCopyMemory( proposal->respSPI, sa->transform[propNum].inSPI,
							sizeof(PGPipsecSPI) );
			pgpCopyMemory( proposal->initSPI, sa->transform[propNum].outSPI,
							sizeof(PGPipsecSPI) );
		}
		switch( proposal->protocol )
		{
			case kPGPike_PR_AH:
				pgpCopyMemory( &proposal->t[0].u.ipsec.ah,
								&sa->transform[propNum].u.ah.t,
								sizeof(PGPipsecAHTransform) );
				break;
			case kPGPike_PR_ESP:
				
				pgpCopyMemory( &proposal->t[0].u.ipsec.esp,
								&sa->transform[propNum].u.esp.t,
								sizeof(PGPipsecESPTransform) );
				break;
			case kPGPike_PR_IPCOMP:
				pgpCopyMemory( &proposal->t[0].u.ipsec.ipcomp,
								&sa->transform[propNum].u.ipcomp.t,
								sizeof(PGPipsecIPCOMPTransform) );
				break;
			default:
				break;
		}
		err = pgpIKEGetP2Keymat( exchange, &sa->transform[propNum] );CKERR;
		propNum++;
	}
	sa->numTransforms = propNum;
	
	sa->nextSA = exchange->ike->sa;
	if( IsntNull( sa->nextSA ) )
		sa->nextSA->prevSA = sa;
	exchange->ike->sa = sa;
	exchange->partner->termSchedule = 0;
	*saP = sa;
done:
	return err;
}

	PGPError
pgpIKEGetAuthHash(
	PGPikeExchange *		exchange,
	PGPBoolean				initiator,
	PGPByte *				outHash )
{
	PGPError				err = kPGPError_NoErr;
	PGPHMACContextRef		hmac;
	PGPUInt32				ySize;
	PGPByte					yData[512];
	
	err = PGPNewHMACContext( exchange->ike->pgpContext,
							exchange->partner->sdkHashAlg,
							&exchange->partner->skeyid[0],
							exchange->partner->hashSize, &hmac );	CKERR;
	ySize = ( ( PGPBigNumGetSignificantBits( exchange->dhP ) + 7 ) / 8 );
	if( initiator )
	{
		err = PGPBigNumExtractBigEndianBytes( exchange->dhYi, yData, 0,
												ySize ); CKERR;
		(void)PGPContinueHMAC( hmac, yData, ySize );
		err = PGPBigNumExtractBigEndianBytes( exchange->dhYr, yData, 0,
												ySize ); CKERR;
		(void)PGPContinueHMAC( hmac, yData, ySize );
		(void)PGPContinueHMAC( hmac, exchange->partner->initCookie,
								kPGPike_CookieSize );
		(void)PGPContinueHMAC( hmac, exchange->partner->respCookie,
								kPGPike_CookieSize );
		(void)PGPContinueHMAC( hmac, exchange->initSABody,
								exchange->initSABodySize );
		(void)PGPContinueHMAC( hmac, exchange->idBody,
								exchange->idBodySize );
	}
	else
	{
		err = PGPBigNumExtractBigEndianBytes( exchange->dhYr, yData, 0, ySize ); CKERR;
		(void)PGPContinueHMAC( hmac, yData, ySize );
		err = PGPBigNumExtractBigEndianBytes( exchange->dhYi, yData, 0, ySize ); CKERR;
		(void)PGPContinueHMAC( hmac, yData, ySize );
		(void)PGPContinueHMAC( hmac, exchange->partner->respCookie,
								kPGPike_CookieSize );
		(void)PGPContinueHMAC( hmac, exchange->partner->initCookie,
								kPGPike_CookieSize );
		(void)PGPContinueHMAC( hmac, exchange->initSABody,
								exchange->initSABodySize );
		(void)PGPContinueHMAC( hmac, exchange->idRBody,
								exchange->idRBodySize );
	}
	
	err = PGPFinalizeHMAC( hmac, outHash );	CKERR;
done:
	if( PGPHMACContextRefIsValid( hmac ) )
		(void)PGPFreeHMACContext( hmac );
	return err;
}

	PGPBoolean
pgpIKEIsTransformEqual(
	PGPipsecProtocol				protocol,
	const PGPikeGenericTransform *	transform1,
	const PGPikeGenericTransform *	transform2 )
{
	PGPBoolean						same = FALSE;
	
	switch( protocol )
	{
		case kPGPike_PR_IKE:
		{	
			PGPikeTransform *		t = (PGPikeTransform *)transform1;
			PGPikeTransform *		t2 = (PGPikeTransform *)transform2;
			
			if( ( t->authMethod == t2->authMethod ) &&
				( t->hash == t2->hash ) &&
				( t->cipher == t2->cipher ) &&
				( t->modpGroup == t2->modpGroup ) &&
				( t->groupID == t2->groupID ) )
			{
				same = TRUE;
			}
			break;
		}
		case kPGPike_PR_ESP:
		{	
			PGPipsecESPTransform *	t = (PGPipsecESPTransform *)transform1;
			PGPipsecESPTransform *	t2 = (PGPipsecESPTransform *)transform2;
			
			if( ( t->cipher == t2->cipher ) &&
				( t->authAttr == t2->authAttr ) &&
				( t->mode == t2->mode ) )
				same = TRUE;
			break;
		}
		case kPGPike_PR_AH:
		{
			PGPipsecAHTransform *	t = (PGPipsecAHTransform *)transform1;
			PGPipsecAHTransform *	t2 = (PGPipsecAHTransform *)transform2;
			
			if( ( t->authAlg == t2->authAlg ) &&
				( t->authAttr == t2->authAttr ) &&
				( t->mode == t2->mode ) )
				same = TRUE;
			break;
		}
		case kPGPike_PR_IPCOMP:
		{	
			PGPipsecIPCOMPTransform *	t = (PGPipsecIPCOMPTransform *)transform1;
			PGPipsecIPCOMPTransform *	t2 = (PGPipsecIPCOMPTransform *)transform2;
			
			if( t->compAlg == t2->compAlg )	/* ***** more attributes? */
				same = TRUE;
			break;
		}
		default:
			break;
	}
	return same;
}

	PGPBoolean
pgpIKEIsTransformValid(
	PGPikePartner *					partner,
	PGPipsecProtocol				protocol,
	const PGPikeGenericTransform *	transform )
{
	PGPBoolean						valid = FALSE,
									prop;
	PGPInt32						alg;
	PGPikeAllowedAlgorithms *		allow = &partner->ike->allowedAlgorithms;
	
	switch( protocol )
	{
		case kPGPike_PR_IKE:
		{
			PGPKeyDBObjRef			key = PGPPeekKeyDBObjKey(partner->pgpAuthKey.authObj),
									certKey = PGPPeekKeyDBObjKey( partner->x509AuthKey.authObj );
			
			switch( transform->u.ike.authMethod )
			{
				case kPGPike_AM_HAuth_InitRSA:
				case kPGPike_AM_HAuth_InitDSS:
					if( !partner->initiator )
						goto done;
				case kPGPike_AM_HAuth_RespRSA:
				case kPGPike_AM_HAuth_RespDSS:
					if( partner->authStyle != kPGPike_AS_HybridAuth )
						goto done;
					break;
#if !PGPIKE_XAUTHCISCO
				case kPGPike_AM_XAuth_InitPreShared:
					if( !partner->initiator )
						goto done;
				case kPGPike_AM_XAuth_RespPreShared:
					if( partner->authStyle != kPGPike_AS_XAuth )
						goto done;
#endif
				case kPGPike_AM_PreSharedKey:
#if !PGPIKE_XAUTHCISCO
					if( ( transform->u.ike.authMethod == kPGPike_AM_PreSharedKey ) &&
						( partner->authStyle == kPGPike_AS_XAuth ) )
						goto done;
#endif
					if( IsNull( partner->sharedKey ) ||
						( partner->sharedKeySize == 0 ) )
						goto done;
					break;
#if !PGPIKE_XAUTHCISCO
				case kPGPike_AM_XAuth_InitDSS:
					if( !partner->initiator )
						goto done;
				case kPGPike_AM_XAuth_RespDSS:
					if( partner->authStyle != kPGPike_AS_XAuth )
						goto done;
#endif
				case kPGPike_AM_DSS_Sig:
#if !PGPIKE_XAUTHCISCO
					if( ( transform->u.ike.authMethod == kPGPike_AM_DSS_Sig ) &&
						( partner->authStyle == kPGPike_AS_XAuth ) )
						goto done;
#endif
					if( PGPKeyDBObjRefIsValid( key ) &&
						IsntPGPError( PGPGetKeyDBObjNumericProperty( key,
										kPGPKeyProperty_AlgorithmID, &alg ) ) &&
						( alg == kPGPPublicKeyAlgorithm_DSA ) &&
						IsntPGPError( PGPGetKeyDBObjBooleanProperty( key,
										kPGPKeyProperty_CanSign, &prop ) ) &&
						prop )
						break;
					if( PGPKeyDBObjRefIsValid( certKey ) &&
						IsntPGPError( PGPGetKeyDBObjNumericProperty( certKey,
										kPGPKeyProperty_AlgorithmID, &alg ) ) &&
						( alg == kPGPPublicKeyAlgorithm_DSA ) &&
						IsntPGPError( PGPGetKeyDBObjBooleanProperty( certKey,
										kPGPKeyProperty_CanSign, &prop ) ) &&
						prop )
						break;
					goto done;
#if !PGPIKE_XAUTHCISCO
				case kPGPike_AM_XAuth_InitRSA:
					if( !partner->initiator )
						goto done;
				case kPGPike_AM_XAuth_RespRSA:
					if( partner->authStyle != kPGPike_AS_XAuth )
						goto done;
#endif
				case kPGPike_AM_RSA_Sig:
#if !PGPIKE_XAUTHCISCO
					if( ( transform->u.ike.authMethod == kPGPike_AM_RSA_Sig ) &&
						( partner->authStyle == kPGPike_AS_XAuth ) )
						goto done;
#endif
					if( PGPKeyDBObjRefIsValid( key ) &&
						IsntPGPError( PGPGetKeyDBObjNumericProperty( key,
										kPGPKeyProperty_AlgorithmID, &alg ) ) &&
						( alg == kPGPPublicKeyAlgorithm_RSA ) &&
						IsntPGPError( PGPGetKeyDBObjBooleanProperty( key,
										kPGPKeyProperty_CanSign, &prop ) ) &&
						prop )
						break;
					if( PGPKeyDBObjRefIsValid( certKey ) &&
						IsntPGPError( PGPGetKeyDBObjNumericProperty( certKey,
										kPGPKeyProperty_AlgorithmID, &alg ) ) &&
						( alg == kPGPPublicKeyAlgorithm_RSA ) &&
						IsntPGPError( PGPGetKeyDBObjBooleanProperty( certKey,
										kPGPKeyProperty_CanSign, &prop ) ) &&
						prop )
						break;
				case kPGPike_AM_RSA_Encrypt:
				case kPGPike_AM_RSA_Encrypt_R:
				case kPGPike_AM_XAuth_InitRSAEncryption:
				case kPGPike_AM_XAuth_RespRSAEncryption:
				case kPGPike_AM_XAuth_InitRSAREncryption:
				case kPGPike_AM_XAuth_RespRSAREncryption:
				default:
					goto done;
			}
			switch( transform->u.ike.hash )
			{
				case kPGPike_HA_SHA1:
					if( !allow->sha1 )
						goto done;
					break;
				case kPGPike_HA_MD5:
					if( !allow->md5 )
						goto done;
					break;
				default:
					goto done;
			}
			switch( transform->u.ike.cipher )
			{
				case kPGPike_SC_CAST_CBC:
					if( !allow->cast5 )
						goto done;
					break;
				case kPGPike_SC_3DES_CBC:
					if( !allow->tripleDES )
						goto done;
					break;
				case kPGPike_SC_DES_CBC:
					if( !allow->singleDES )
						goto done;
					break;
				default:
					goto done;
			}
			if( !transform->u.ike.modpGroup )
				goto done;
			switch( transform->u.ike.groupID )
			{
				case kPGPike_GR_MODPOne:
					if( !allow->modpOne768 )
						goto done;
					break;
				case kPGPike_GR_MODPTwo:
					if( !allow->modpTwo1024 )
						goto done;
					break;
				case kPGPike_GR_MODPFive:
					if( !allow->modpFive1536 )
						goto done;
					break;
				default:
					goto done;
			}
			/* This transform looks good */
			valid = TRUE;
			break;
		}
		case kPGPike_PR_AH:
			switch( transform->u.ipsec.ah.authAlg )
			{
				case kPGPike_AH_MD5:
					if( !allow->md5 )
						goto done;
					break;
				case kPGPike_AH_SHA:
					if( !allow->sha1 )
						goto done;
					break;
				default:
					goto done;
			}
			switch( transform->u.ipsec.ah.authAttr )
			{
				case kPGPike_AA_HMAC_MD5:
					if( transform->u.ipsec.ah.authAlg != kPGPike_AH_MD5 )
						goto done;
					break;
				case kPGPike_AA_HMAC_SHA:
					if( transform->u.ipsec.ah.authAlg != kPGPike_AH_SHA )
						goto done;
					break;
				default:
					goto done;
			}
			switch( transform->u.ipsec.ah.mode )
			{
				case kPGPike_PM_Tunnel:
				case kPGPike_PM_Transport:
					break;
				default:
					goto done;
			}
			switch( transform->u.ipsec.groupID )
			{
				case kPGPike_GR_None:
					break;
				case kPGPike_GR_MODPOne:
					if( !allow->modpOne768 )
						goto done;
					break;
				case kPGPike_GR_MODPTwo:
					if( !allow->modpTwo1024 )
						goto done;
					break;
				case kPGPike_GR_MODPFive:
					if( !allow->modpFive1536 )
						goto done;
					break;
				default:
					goto done;
			}
			/* This transform looks good */
			valid = TRUE;
			break;
		case kPGPike_PR_ESP:
			switch( transform->u.ipsec.esp.cipher )
			{
				case kPGPike_ET_3DES:
					if( !allow->tripleDES )
						goto done;
					break;
				case kPGPike_ET_CAST:
					if( !allow->cast5 )
						goto done;
					break;
				case kPGPike_ET_DES:
				case kPGPike_ET_DES_IV64:
					if( !allow->singleDES )
						goto done;
					break;
				case kPGPike_ET_NULL:
					if( !allow->espNULL )
						goto done;
					break;
				default:
					goto done;
			}
			switch( transform->u.ipsec.esp.authAttr )
			{
				case kPGPike_AA_HMAC_MD5:
					if( !allow->md5 )
						goto done;
					break;
				case kPGPike_AA_HMAC_SHA:
					if( !allow->sha1 )
						goto done;
					break;
				case kPGPike_AA_None:
					if( !allow->noAuth )
						goto done;
					break;
				default:
					goto done;
			}
			if( ( transform->u.ipsec.esp.authAttr == kPGPike_AA_None ) &&
				( transform->u.ipsec.esp.cipher == kPGPike_ET_NULL ) )
				goto done;
			switch( transform->u.ipsec.esp.mode )
			{
				case kPGPike_PM_Tunnel:
				case kPGPike_PM_Transport:
					break;
				default:
					goto done;
			}
			switch( transform->u.ipsec.groupID )
			{
				case kPGPike_GR_None:
					break;
				case kPGPike_GR_MODPOne:
					if( !allow->modpOne768 )
						goto done;
					break;
				case kPGPike_GR_MODPTwo:
					if( !allow->modpTwo1024 )
						goto done;
					break;
				case kPGPike_GR_MODPFive:
					if( !allow->modpFive1536 )
						goto done;
					break;
				default:
					goto done;
			}
			/* This transform looks good */
			valid = TRUE;
			break;
		case kPGPike_PR_IPCOMP:
			switch( transform->u.ipsec.ipcomp.compAlg )
			{
				case kPGPike_IC_Deflate:
					if( !allow->deflate )
						goto done;
					break;
				case kPGPike_IC_LZS:
					if( !allow->lzs )
						goto done;
					break;
				default:
					goto done;
			}
			/* This transform looks good */
			valid = TRUE;
			break;
		default:
			goto done;
	}
done:
	return valid;
}

	PGPError
pgpIKEResendLastPacket(
	PGPikeExchange *		exchange )
{
	PGPError				err = kPGPError_NoErr;

#if PGPIKE_DEBUG
	pgpIKEDebugLog( exchange->ike, TRUE, "No Response - Resent last packet (%d tries remaining)\n",
		exchange->retry - 1 );
#endif
	--exchange->retry;
	if( !exchange->retry )
	{
		err = pgpIKEAbortExchange( &exchange, kPGPike_AL_None );
		goto done;
	}
	exchange->lastTransmit	= PGPGetMilliseconds();
	err = pgpIKESendPacket1( exchange->partner, exchange->lastPkt,
								exchange->lastPktSize );
done:
	return err;
}

/* send packet and place it in retry position */
	PGPError
pgpIKESendPacket(
	PGPikeExchange *		exchange,
	PGPByte *				packet,
	PGPSize					packetSize )
{
	PGPError				err = kPGPError_NoErr;
	
	pgpAssertAddrValid( exchange, PGPikeExchange * );

#if PGPIKE_DEBUG
	pgpIKEDebugLog( exchange->ike, FALSE, "SENT\n" );
#endif
	err = pgpIKESendPacket1( exchange->partner, packet, packetSize );
	
	/* set packet up for future retry */
	if( IsntNull( exchange->lastPkt ) )
		(void)PGPFreeData( exchange->lastPkt );
	
	exchange->retry			= kPGPike_MaxRetries;
	exchange->lastTransmit	= PGPGetMilliseconds();
	exchange->lastPkt		= packet;
	exchange->lastPktSize	= packetSize;
	
	return err;
}

/* Send the header-equipped packet to the caller */
	PGPError
pgpIKESendPacket1(
	PGPikePartner *			partner,
	PGPByte *				packet,
	PGPSize					packetSize )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeMTPacket			s;
	
	pgpAssertAddrValid( partner, PGPikePartner * );
	if( IsNull( packet ) || !packetSize )
		return err;
	
	s.ipAddress			= partner->ipAddress;
	s.port				= partner->destPort;
	s.packetSize		= packetSize;
	s.packet			= packet;
	
	err = (partner->ike->msgProc)( (PGPikeContextRef)partner->ike,
									partner->ike->userData,
									kPGPike_MT_Packet, &s );
	
	return err;
}

	PGPError
pgpIKEGetCert(
	PGPikePartner *			partner,
	PGPikeMessageType		msg )
{
	PGPError				err = kPGPError_NoErr;
	PGPikeMTCert			cert;
	PGPKeyDBObjType			objType;
	
	pgpAssertAddrValid( partner, PGPikePartner * );
	pgpClearMemory( &cert, sizeof(PGPikeMTCert) );
	cert.ipAddress = partner->ipAddress;
	err = (partner->ike->msgProc)( (PGPikeContextRef)partner->ike,
									partner->ike->userData,
									msg, &cert );
	
	if( IsntPGPError( err ) && PGPKeyDBObjRefIsValid( cert.authObj ) )
	{
		void *	callerPass = cert.pass;
		
		err = PGPGetKeyDBObjNumericProperty( cert.authObj,
				kPGPKeyDBObjProperty_ObjectType, (PGPInt32 *)&objType );
		if( ( msg == kPGPike_MT_LocalX509Cert ) &&
			( objType != kPGPKeyDBObjType_Signature ) )
		{
			err = kPGPError_BadParams;
			goto done;
		}
		else if( ( msg == kPGPike_MT_LocalPGPCert ) &&
			( objType != kPGPKeyDBObjType_Key ) )
		{
			err = kPGPError_BadParams;
			goto done;
		}
		if( PGPKeyDBRefIsValid( cert.baseKeyDB ) )
			PGPIncKeyDBRefCount( cert.baseKeyDB );
		if( IsntNull( callerPass ) )
		{
			if( cert.passLength == 0 )
			{
				cert.isPassKey = FALSE;
				cert.passLength = 1;
				callerPass = "\0";
			}
			cert.pass = PGPNewSecureData( partner->ike->memMgr, cert.passLength,
							kPGPMemoryMgrFlags_Clear );
			if( IsntNull( cert.pass ) )
				pgpCopyMemory( callerPass, cert.pass, cert.passLength );
			else
				err = kPGPError_OutOfMemory;
		}
		else
		{
			cert.isPassKey = FALSE;
			cert.passLength = 0;
			cert.pass = NULL;
		}
		switch( msg )
		{
			case kPGPike_MT_LocalPGPCert:
				pgpCopyMemory( &cert, &partner->pgpAuthKey,
								sizeof( PGPikeMTCert ) );
				break;
			case kPGPike_MT_LocalX509Cert:
				pgpCopyMemory( &cert, &partner->x509AuthKey,
								sizeof( PGPikeMTCert ) );
				break;
			default:
				pgpAssert( 0 );
				break;
		}
	}
done:
	return err;
}

	void
pgpIKEDebugData(
	PGPikeContextPriv *		ike,
	char *					name,
	PGPByte *				data,
	PGPSize					length )
{
	char					buffer[1024];
	PGPUInt16				bIndex,
							bufLen;
	
	if( length > sizeof(buffer) / 5 )
		length = sizeof(buffer) / 5;
	for( bufLen = bIndex = 0; bIndex < length; bIndex++, bufLen += 4 )
	{
		sprintf( buffer + bufLen, "%02X%s", data[bIndex],
					( ( bIndex + 1 ) % 16 ) ? "  " : "\n\t" );
	}
	pgpIKEDebugLog( ike, FALSE, "\t%s (%d):\n\t%s\n", name, length, buffer );
}

	void
pgpIKEDebugLog(
	PGPikeContextPriv *		ike,
	PGPBoolean				timer,
	char *					fmt,
	... )
{
	va_list					ap;
	char					buf[1024];
	PGPUInt32				curTime,
							hours,
							minutes,
							seconds,
							remain;
	
	buf[0] = '\0';
	if( timer )
	{
		curTime = PGPGetMilliseconds();
		hours = curTime / ( 1000 * 60 * 60 );
		remain =  curTime - ( hours * ( 1000 * 60 * 60 ) );
		minutes = remain / ( 1000 * 60 );
		seconds = ( remain - ( minutes * ( 1000 * 60 ) ) ) / 1000;
		sprintf( buf, "%02u:%02u:%02u: ", hours, minutes, seconds );
	}
	va_start( ap, fmt );
	vsprintf( buf + strlen(buf), fmt, ap );
	va_end( ap );

	(void)(ike->msgProc)( (PGPikeContextRef)ike,
							ike->userData,
							kPGPike_MT_DebugLog, buf );
}

	char *
pgpIKECipherString(
	PGPikeCipher			cipher )
{
	char	*				ciStr = "unknown";
	
	switch( cipher )
	{
		case kPGPike_SC_DES_CBC:	ciStr = "SingleDES";	break;
		case kPGPike_SC_3DES_CBC:	ciStr = "TripleDES";	break;
		case kPGPike_SC_CAST_CBC:	ciStr = "CAST5";		break;
		default:
			break;
	}
	return ciStr;
}

	char *
pgpIKEAuthMethodString(
	PGPikeAuthMethod		auth )
{
	char	*				amStr = "unknown";
	
	switch( auth )
	{
		case kPGPike_AM_PreSharedKey:				amStr = "PreSharedKey";		break;
		case kPGPike_AM_DSS_Sig:					amStr = "DSS Sig";			break;
		case kPGPike_AM_RSA_Sig:					amStr = "RSA Sig";			break;
		case kPGPike_AM_RSA_Encrypt:				amStr = "RSA Encrypt";		break;
		case kPGPike_AM_RSA_Encrypt_R:				amStr = "RSA Encrypt R";	break;
		
		case kPGPike_AM_XAuth_InitPreShared:		amStr = "X-I-PreSharedKey";	break;
		case kPGPike_AM_XAuth_RespPreShared:		amStr = "X-R-PreSharedKey";	break;
		case kPGPike_AM_XAuth_InitDSS:				amStr = "X-I-DSS Sig";		break;
		case kPGPike_AM_XAuth_RespDSS:				amStr = "X-R-DSS Sig";		break;
		case kPGPike_AM_XAuth_InitRSA:				amStr = "X-I-RSA Sig";		break;
		case kPGPike_AM_XAuth_RespRSA:				amStr = "X-R-RSA Sig";		break;
		case kPGPike_AM_XAuth_InitRSAEncryption:	amStr = "X-I-RSA Encrypt";	break;
		case kPGPike_AM_XAuth_RespRSAEncryption:	amStr = "X-R-RSA Encrypt";	break;
		case kPGPike_AM_XAuth_InitRSAREncryption:	amStr = "X-I-RSA Encrypt R";break;
		case kPGPike_AM_XAuth_RespRSAREncryption:	amStr = "X-R-RSA Encrypt R";break;
		case kPGPike_AM_HAuth_InitRSA:				amStr = "H-I-RSA Sig";		break;
		case kPGPike_AM_HAuth_RespRSA:				amStr = "H-R-RSA Sig";		break;
		case kPGPike_AM_HAuth_InitDSS:				amStr = "H-I-DSS Sig";		break;
		case kPGPike_AM_HAuth_RespDSS:				amStr = "H-R-DSS Sig";		break;
		default:
			break;
	}
	return amStr;
}

	char *
pgpIKEProtocolString(
	PGPipsecProtocol		protocol )
{
	char	*				prStr = "unknown";
	
	switch( protocol )
	{
		case kPGPike_PR_IKE:			prStr = "IKE";			break;
		case kPGPike_PR_AH:				prStr = "AH";			break;
		case kPGPike_PR_ESP:			prStr = "ESP";			break;
		case kPGPike_PR_IPCOMP:			prStr = "IPCOMP";		break;
		default:
			break;
	}
	return prStr;
}

	char *
pgpIKEPayloadTypeString(
	PGPikePayload			payload )
{
	char	*				ptStr = "unknown";
	
	switch( payload )
	{
		case kPGPike_PY_SA:				ptStr = "SA";				break;
		case kPGPike_PY_Proposal:		ptStr = "Prop";				break;
		case kPGPike_PY_Transform:		ptStr = "Trans";			break;
		case kPGPike_PY_KeyExchange:	ptStr = "KE";				break;
		case kPGPike_PY_Identification:	ptStr = "Ident";			break;
		case kPGPike_PY_Certificate:	ptStr = "Cert";				break;
		case kPGPike_PY_CertRequest:	ptStr = "CertReq";			break;
		case kPGPike_PY_Hash:			ptStr = "Hash";				break;
		case kPGPike_PY_Signature:		ptStr = "Sig";				break;
		case kPGPike_PY_Nonce:			ptStr = "Nonce";			break;
		case kPGPike_PY_Notification:	ptStr = "Notify";			break;
		case kPGPike_PY_Delete:			ptStr = "Delete";			break;
		case kPGPike_PY_VendorID:		ptStr = "Vendor";			break;
		case kPGPike_PY_Attribute:		ptStr = "Attr";				break;
		case kPGPike_PY_NATDiscovery:   ptStr = "NATDisc";			break;
		default:
			break;
	}
	return ptStr;
}

	char *
pgpIKEExchangeTypeString(
	PGPikeExchangeT			exchange )
{
	char	*				etStr = "unknown";
	
	switch( exchange )
	{
		case kPGPike_EX_Base:			etStr = "Base";				break;
		case kPGPike_EX_Identity:		etStr = "Identity";			break;
		case kPGPike_EX_Authentication:	etStr = "Auth";				break;
		case kPGPike_EX_Aggressive:		etStr = "Aggressive";		break;
		case kPGPike_EX_Informational:	etStr = "Informational";	break;
		case kPGPike_EX_Transaction:	etStr = "Transaction";		break;
		case kPGPike_EX_IPSEC_Quick:	etStr = "Quick";			break;
		case kPGPike_EX_IPSEC_NewGroup:	etStr = "NewGroup";			break;
		default:
			break;
	}
	return etStr;
}

	char *
pgpIKEP1AttributeString(
	PGPikeAttributeType		attribute )
{
	char	*				atStr = "unknown";
	
	switch( attribute )
	{
		case kPGPike_AT_EncryptAlg:				atStr = "EncryptAlg";		break;
		case kPGPike_AT_HashAlg:				atStr = "HashAlg";			break;
		case kPGPike_AT_AuthMethod:				atStr = "AuthMethod";		break;
		case kPGPike_AT_GroupDesc:				atStr = "GroupDesc";		break;
		case kPGPike_AT_GroupType:				atStr = "GroupType";		break;
		case kPGPike_AT_GroupPrime:				atStr = "GroupPrime";		break;
		case kPGPike_AT_GroupGenOne:			atStr = "GroupGenOne";		break;
		case kPGPike_AT_GroupGenTwo:			atStr = "GroupGenTwo";		break;
		case kPGPike_AT_GroupCurveA:			atStr = "GroupCurveA";		break;
		case kPGPike_AT_GroupCurveB:			atStr = "GroupCurveB";		break;
		case kPGPike_AT_LifeType:				atStr = "LifeType";			break;
		case kPGPike_AT_LifeDuration:			atStr = "LifeDuration";		break;
		case kPGPike_AT_PRF:					atStr = "PRF";				break;
		case kPGPike_AT_KeyLength:				atStr = "KeyLength";		break;
		case kPGPike_AT_FieldSize:				atStr = "FieldSize";		break;
		case kPGPike_AT_GroupOrder:				atStr = "GroupOrder";		break;
		default:
			break;
	}
	return atStr;
}

	char *
pgpIKEP2AttributeString(
	PGPikeAttributeType		attribute )
{
	char	*				atStr = "unknown";
	
	switch( attribute )
	{
		case kPGPike_AT_IPSEC_LifeType:			atStr = "ILifeType";		break;
		case kPGPike_AT_IPSEC_Duration:			atStr = "IDuration";		break;
		case kPGPike_AT_IPSEC_GroupDesc:		atStr = "IGroupDesc";		break;
		case kPGPike_AT_IPSEC_Encapsulation:	atStr = "Encapsulation";	break;
		case kPGPike_AT_IPSEC_AuthAttr:			atStr = "AuthAttr";			break;
		case kPGPike_AT_IPSEC_KeyLength:		atStr = "IKeyLength";		break;
		case kPGPike_AT_IPSEC_KeyRounds:		atStr = "KeyRounds";		break;
		case kPGPike_AT_IPSEC_CompressSize:		atStr = "CompressSize";		break;
		case kPGPike_AT_IPSEC_CompressAlg:		atStr = "CompressAlg";		break;
		default:
			break;
	}
	return atStr;
}

	char *
pgpIKEConfigAttrString(
	PGPikeAttributeType		attribute )
{
	char	*				atStr = "unknown";
	
	switch( attribute )
	{
		case kPGPike_AT_Config_IPv4Address:		atStr = "ipv4Addr";		break;
		case kPGPike_AT_Config_IPv4Mask:		atStr = "ipv4Mask";		break;
		case kPGPike_AT_Config_IPv4DNS:			atStr = "ipv4DNS";		break;
		case kPGPike_AT_Config_IPv4NBNS:		atStr = "ipv4NBNS";		break;
		case kPGPike_AT_Config_AddressExpiry:	atStr = "AddrExpiry";	break;
		case kPGPike_AT_Config_IPv4DHCP:		atStr = "ipv4DHCP";		break;
		case kPGPike_AT_Config_AppVersion:		atStr = "AppVersion";	break;
		case kPGPike_AT_Config_IPv6Address:		atStr = "ipv6Addr";		break;
		case kPGPike_AT_Config_IPv6Mask:		atStr = "ipv6Mask";		break;
		case kPGPike_AT_Config_IPv6DNS:			atStr = "ipv6DNS";		break;
		case kPGPike_AT_Config_IPv6NBNS:		atStr = "ipv6NBNS";		break;
		case kPGPike_AT_Config_IPv6DHCP:		atStr = "ipv6DHCP";		break;
		
		case kPGPike_AT_XAuth_Type:				atStr = "xauthType";	break;
		case kPGPike_AT_XAuth_Username:			atStr = "xauthUsername";break;
		case kPGPike_AT_XAuth_Password:			atStr = "xauthPassword";break;
		case kPGPike_AT_XAuth_Passcode:			atStr = "xauthPasscode";break;
		case kPGPike_AT_XAuth_Message:			atStr = "xauthMessage";	break;
		case kPGPike_AT_XAuth_Challenge:		atStr = "xauthChallenge";break;
		case kPGPike_AT_XAuth_Domain:			atStr = "xauthDomain";	break;
		case kPGPike_AT_XAuth_Status:			atStr = "xauthStatus";	break;
		default:
			break;
	}
	return atStr;
}

	char *
pgpIKEIDTypeString(
	PGPipsecIdentity		id )
{
	char *					idStr = "unknown";
	
	switch( id )
	{
		case kPGPike_ID_IPV4_Addr:			idStr = "IPv4 Addr";	break;
		case kPGPike_ID_FQDN:				idStr = "FQDN";			break;
		case kPGPike_ID_UserFQDN:			idStr = "UserFQDN";		break;
		case kPGPike_ID_IPV4_Addr_Subnet:	idStr = "IPv4 Subnet";	break;
		case kPGPike_ID_IPV6_Addr:			idStr = "IPv6 Addr";	break;
		case kPGPike_ID_IPV6_Addr_Subnet:	idStr = "IPv6 Subnet";	break;
		case kPGPike_ID_IPV4_Addr_Range:	idStr = "IPv4 Range";	break;
		case kPGPike_ID_IPV6_Addr_Range:	idStr = "IPv6 Range";	break;
		case kPGPike_ID_DER_ASN1_DN:		idStr = "DER DN";		break;
		case kPGPike_ID_DER_ASN1_GN:		idStr = "DER GN";		break;
		case kPGPike_ID_Key_ID:				idStr = "KeyID";		break;
		default:
			break;
	}
	return idStr;
}

	char *
pgpIKEipsecCipherTypeString(
	PGPipsecESPTransformID	cipher )
{
	char *					cipherStr = "unknown";
	
	switch( cipher )
	{
		case kPGPike_ET_DES_IV64:			cipherStr = "DES-IV64";		break;
		case kPGPike_ET_DES:				cipherStr = "DES";			break;
		case kPGPike_ET_3DES:				cipherStr = "3DES";			break;
		case kPGPike_ET_RC5:				cipherStr = "RC5";			break;
		case kPGPike_ET_IDEA:				cipherStr = "IDEA";			break;
		case kPGPike_ET_CAST:				cipherStr = "CAST";			break;
		case kPGPike_ET_Blowfish:			cipherStr = "Blowfish";		break;
		case kPGPike_ET_3IDEA:				cipherStr = "3IDEA";		break;
		case kPGPike_ET_DES_IV32:			cipherStr = "DES-IV32";		break;
		case kPGPike_ET_RC4:				cipherStr = "RC4";			break;
		case kPGPike_ET_NULL:				cipherStr = "NULL";			break;
		case kPGPike_ET_AES:				cipherStr = "AES";			break;
		default:
			break;
	}
	return cipherStr;
}

	char *
pgpIKEipsecAuthTypeString(
	PGPipsecAuthAttribute	auth )
{
	char *					authStr = "unknown";
	
	switch( auth )
	{
		case kPGPike_AA_None:				authStr = "None";			break;
		case kPGPike_AA_HMAC_MD5:			authStr = "MD5-HMAC";		break;
		case kPGPike_AA_HMAC_SHA:			authStr = "SHA-HMAC";		break;
		case kPGPike_AA_DES_MAC:			authStr = "DES-MAC";		break;
		case kPGPike_AA_KPDK:				authStr = "KPDK";			break;
		case kPGPike_AA_HMAC_SHA2_256:		authStr = "SHA2-256-HMAC";	break;
		case kPGPike_AA_HMAC_SHA2_384:		authStr = "SHA2-384-HMAC";	break;
		case kPGPike_AA_HMAC_SHA2_512:		authStr = "SHA2-512-HMAC";	break;
		default:
			break;
	}
	return authStr;
}

	char *
pgpIKEHashTypeString(
	PGPikeHash				hash )
{
	char *					hashStr = "unknown";
	
	switch( hash )
	{
		case kPGPike_HA_MD5:				hashStr	= "MD5";		break;
		case kPGPike_HA_SHA1:				hashStr	= "SHA";		break;
		case kPGPike_HA_Tiger:				hashStr	= "Tiger";		break;
		case kPGPike_HA_SHA2_256:			hashStr	= "SHA2-256";	break;
		case kPGPike_HA_SHA2_384:			hashStr	= "SHA2-384";	break;
		case kPGPike_HA_SHA2_512:			hashStr	= "SHA2-512";	break;
		default:
			break;
	}
	return hashStr;
}

	char *
pgpIKEGroupTypeString(
	PGPikeGroupID			group )
{
	char *					groupStr = "unknown";
	
	switch( group )
	{
		case kPGPike_GR_MODPOne:			groupStr = "modP-768";	break;
		case kPGPike_GR_MODPTwo:			groupStr = "modP-1024";	break;
		case kPGPike_GR_MODPFive:			groupStr = "modP-1536";	break;
		default:
			break;
	}
	return groupStr;
}

	char *
pgpIKEGetAlertString(
	PGPikeAlert			alert,
	PGPikeInternalAlert	ia )
{
	char *				alertStr = "unknown";
	
	switch( alert )
	{
		case kPGPike_AL_None:
			switch( ia )
			{
				case kPGPike_IA_ResponseTimeout:	alertStr = "ResponseTimeout";	break;
				case kPGPike_IA_NoProposals:		alertStr = "NoProposals";		break;
				case kPGPike_IA_NewPhase1SA:		alertStr = "NewPhase1SA";		break;
				case kPGPike_IA_NewPhase2SA:		alertStr = "NewPhase2SA";		break;
				case kPGPike_IA_DeadPhase1SA:		alertStr = "DeadPhase1SA";		break;
				case kPGPike_IA_DeadPhase2SA:		alertStr = "DeadPhase2SA";		break;
				case kPGPike_IA_TooManyExchanges:	alertStr = "TooManyExchanges";	break;
				case kPGPike_IA_XAuthSuccess:		alertStr = "XAuthSuccess";		break;
				case kPGPike_IA_XAuthFailed:		alertStr = "XAuthFailed";		break;
				default:
					break;
			}
			break;
		case kPGPike_AL_InvalidPayload:			alertStr = "InvalidPayload";		break;
		case kPGPike_AL_DOIUnsupported:			alertStr = "DOIUnsupported";		break;
		case kPGPike_AL_SituationUnsupported:	alertStr = "SituationUnsupported";	break;
		case kPGPike_AL_InvalidCookie:			alertStr = "InvalidCookie";			break;
		case kPGPike_AL_InvalidMajorVersion:	alertStr = "InvalidMajorVersion";	break;
		case kPGPike_AL_InvalidMinorVersion:	alertStr = "InvalidMinorVersion";	break;
		case kPGPike_AL_InvalidExchange:		alertStr = "InvalidExchange";		break;
		case kPGPike_AL_InvalidFlags:			alertStr = "InvalidFlags";			break;
		case kPGPike_AL_InvalidMessageID:		alertStr = "InvalidMessageID";		break;
		case kPGPike_AL_InvalidProtocolID:		alertStr = "InvalidProtocolID";		break;
		case kPGPike_AL_InvalidSPI:				alertStr = "InvalidSPI";			break;
		case kPGPike_AL_InvalidTransform:		alertStr = "InvalidTransform";		break;
		case kPGPike_AL_InvalidAttribute:		alertStr = "InvalidAttribute";		break;
		case kPGPike_AL_NoProposalChoice:		alertStr = "NoProposalChoice";		break;
		case kPGPike_AL_BadProposal:			alertStr = "BadProposal";			break;
		case kPGPike_AL_PayloadMalformed:		alertStr = "PayloadMalformed";		break;
		case kPGPike_AL_InvalidKey:				alertStr = "InvalidKey";			break;
		case kPGPike_AL_InvalidID:				alertStr = "InvalidID";				break;
		case kPGPike_AL_InvalidCertEncoding:	alertStr = "InvalidCertEncoding";	break;
		case kPGPike_AL_InvalidCert:			alertStr = "InvalidCert";			break;
		case kPGPike_AL_UnsupportedCert:		alertStr = "UnsupportedCert";		break;
		case kPGPike_AL_InvalidCertAuthority:	alertStr = "InvalidCertAuthority";	break;
		case kPGPike_AL_InvalidHash:			alertStr = "InvalidHash";			break;
		case kPGPike_AL_AuthenticationFailed:	alertStr = "AuthenticationFailed";	break;
		case kPGPike_AL_InvalidSignature:		alertStr = "InvalidSignature";		break;
		case kPGPike_AL_AddressNotification:	alertStr = "AddressNotification";	break;
		case kPGPike_AL_NotifySALifetime:		alertStr = "NotifySALifetime";		break;
		case kPGPike_AL_CertUnavailable:		alertStr = "CertUnavailable";		break;
		case kPGPike_AL_UnsupportedExchange:	alertStr = "UnsupportedExchange";	break;
		case kPGPike_AL_UnequalPayloadLengths:	alertStr = "UnequalPayloadLengths";	break;
		case kPGPike_AL_Connected:				alertStr = "Connected";				break;
		case kPGPike_AL_ResponderLifetime:		alertStr = "ResponderLifetime";		break;
		case kPGPike_AL_ReplayStatus:			alertStr = "ReplayStatus";			break;
		case kPGPike_AL_InitialContact:			alertStr = "InitialContact";		break;
		default:
			break;
	}
	return alertStr;
}

	void
pgpIKEGetIPString(
	PGPUInt32	ipAddress,
	char *		ipString )
{
	PGPByte *	ipByte = (PGPByte *)&ipAddress;
	
	sprintf( ipString, "%u.%u.%u.%u",
		ipByte[0], ipByte[1], ipByte[2], ipByte[3] );
}


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
